<?php
/**
 * Configuração automática de domínio
 * Detecta se está em desenvolvimento ou produção
 */

// Função para detectar ambiente
function getEnvironment() {
    $host = $_SERVER['HTTP_HOST'] ?? '';
    
    // Ambientes de desenvolvimento
    $devHosts = [
        'localhost',
        '127.0.0.1',
        '::1',
        'localhost:8080',
        '127.0.0.1:8080'
    ];
    
    // IPs locais (desenvolvimento)
    if (preg_match('/^192\.168\./', $host) || preg_match('/^10\./', $host) || preg_match('/^172\.(1[6-9]|2[0-9]|3[01])\./', $host)) {
        return 'development';
    }
    
    // Hosts locais
    if (in_array($host, $devHosts)) {
        return 'development';
    }
    
    // Produção (protecaonarede.com.br)
    if (strpos($host, 'protecaonarede.com.br') !== false) {
        return 'production';
    }
    
    // Default para desenvolvimento
    return 'development';
}

// Configurar URLs baseado no ambiente
function getBaseUrl() {
    $env = getEnvironment();
    $requestUri = $_SERVER['REQUEST_URI'] ?? '';
    
    // Detectar se está dentro de acesso_restrito
    $isInAcessoRestrito = strpos($requestUri, '/acesso_restrito/') !== false;
    
    if ($env === 'production') {
        $baseUrl = 'https://protecaonarede.com.br';
        // Se estiver em acesso_restrito, adicionar ao base URL
        return $isInAcessoRestrito ? $baseUrl . '/acesso_restrito' : $baseUrl;
    } else {
        // Desenvolvimento - usar o host atual
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost:8080';
        $baseUrl = $protocol . '://' . $host;
        // Se estiver em acesso_restrito, adicionar ao base URL
        return $isInAcessoRestrito ? $baseUrl . '/protecaonarede/acesso_restrito' : $baseUrl;
    }
}

// Definir constantes globais
define('ENVIRONMENT', getEnvironment());
define('BASE_URL', getBaseUrl());
define('IS_PRODUCTION', ENVIRONMENT === 'production');
define('IS_DEVELOPMENT', ENVIRONMENT === 'development');

// Configurações específicas do ambiente
if (IS_PRODUCTION) {
    // Produção - configurações otimizadas
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    error_reporting(E_ERROR | E_WARNING | E_PARSE);
} else {
    // Desenvolvimento - debug habilitado
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
}

// Headers de cache baseado no ambiente
function setEnvironmentHeaders() {
    if (IS_PRODUCTION) {
        // Cache agressivo em produção
        header('Cache-Control: public, max-age=3600');
        header('Expires: ' . gmdate('D, d M Y H:i:s \G\M\T', time() + 3600));
    } else {
        // Sem cache em desenvolvimento
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
    }
}

// Aplicar headers automaticamente
setEnvironmentHeaders();

// Função helper para URLs
function url($path = '') {
    return BASE_URL . '/' . ltrim($path, '/');
}

// Função helper para assets
function asset($path) {
    return BASE_URL . '/' . ltrim($path, '/');
}
?>
