/**
 * SISTEMA DE MODAIS PARA CYBERNEWS
 * Modais de compra, interações e feedbacks
 */

(function() {
    'use strict';

    // Criar estrutura base do modal
    function createModalStructure() {
        if (document.getElementById('cybernews-modal')) return;

        const modalHTML = `
        <div id="cybernews-modal" class="modal-overlay" style="display: none;">
            <div class="modal-container">
                <div class="modal-header">
                    <h3 class="modal-title"></h3>
                    <button class="modal-close">&times;</button>
                </div>
                <div class="modal-body"></div>
                <div class="modal-footer"></div>
            </div>
        </div>`;

        document.body.insertAdjacentHTML('beforeend', modalHTML);
        addModalStyles();
        bindModalEvents();
    }

    // Estilos do modal
    function addModalStyles() {
        const style = document.createElement('style');
        style.textContent = `
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            backdrop-filter: blur(10px);
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .modal-overlay.show {
            opacity: 1;
            visibility: visible;
        }

        .modal-container {
            background: rgba(13, 30, 17, 0.95);
            border: 2px solid rgba(144, 238, 144, 0.5);
            border-radius: 20px;
            max-width: 500px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            position: relative;
            box-shadow: 
                0 20px 40px rgba(0, 0, 0, 0.5),
                0 0 30px rgba(144, 238, 144, 0.3);
            transform: scale(0.8);
            transition: all 0.3s ease;
        }

        .modal-overlay.show .modal-container {
            transform: scale(1);
        }

        .modal-header {
            padding: 25px 30px 15px;
            border-bottom: 1px solid rgba(144, 238, 144, 0.3);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-title {
            color: #90ee90;
            margin: 0;
            font-size: 1.5rem;
            text-shadow: 0 0 10px rgba(144, 238, 144, 0.5);
        }

        .modal-close {
            background: none;
            border: none;
            color: #90ee90;
            font-size: 2rem;
            cursor: pointer;
            padding: 0;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: all 0.3s ease;
        }

        .modal-close:hover {
            background: rgba(144, 238, 144, 0.2);
            transform: rotate(90deg);
        }

        .modal-body {
            padding: 20px 30px;
            color: rgba(255, 255, 255, 0.9);
            line-height: 1.6;
        }

        .modal-footer {
            padding: 15px 30px 25px;
            display: flex;
            gap: 15px;
            justify-content: center;
            border-top: 1px solid rgba(144, 238, 144, 0.2);
        }

        .modal-btn {
            padding: 12px 25px;
            border: none;
            border-radius: 25px;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .modal-btn.primary {
            background: linear-gradient(135deg, #2d5016, #4a7c23);
            color: #90ee90;
            border: 2px solid rgba(144, 238, 144, 0.4);
        }

        .modal-btn.primary:hover {
            background: linear-gradient(135deg, #4a7c23, #90ee90);
            color: #0f0f23;
            transform: translateY(-2px);
        }

        .modal-btn.secondary {
            background: rgba(144, 238, 144, 0.1);
            color: #90ee90;
            border: 2px solid rgba(144, 238, 144, 0.3);
        }

        .modal-btn.secondary:hover {
            background: rgba(144, 238, 144, 0.2);
            border-color: rgba(144, 238, 144, 0.6);
        }
        
        @keyframes bounce {
            0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
            40% { transform: translateY(-10px); }
            60% { transform: translateY(-5px); }
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 1; }
            50% { transform: scale(1.05); opacity: 0.8; }
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        @keyframes downloadProgress {
            0% { width: 0%; }
            20% { width: 15%; }
            40% { width: 35%; }
            60% { width: 60%; }
            80% { width: 85%; }
            100% { width: 100%; }
        }
        
        @keyframes connectionProgress {
            0% { width: 0%; }
            25% { width: 30%; }
            50% { width: 55%; }
            75% { width: 80%; }
            100% { width: 100%; }
        }

        .payment-options {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 15px;
            margin: 20px 0;
        }

        .payment-option {
            background: rgba(45, 80, 22, 0.6);
            border: 2px solid rgba(144, 238, 144, 0.3);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .payment-option:hover {
            border-color: rgba(144, 238, 144, 0.8);
            background: rgba(45, 80, 22, 0.8);
            transform: translateY(-3px);
        }

        .payment-option.selected {
            border-color: #90ee90;
            background: rgba(144, 238, 144, 0.2);
        }

        .payment-icon {
            font-size: 2rem;
            margin-bottom: 10px;
            display: block;
        }

        .payment-name {
            font-weight: bold;
            color: #90ee90;
        }
        `;
        document.head.appendChild(style);
    }

    // Eventos do modal
    function bindModalEvents() {
        const modal = document.getElementById('cybernews-modal');
        const closeBtn = modal.querySelector('.modal-close');

        // Fechar modal
        closeBtn.addEventListener('click', closeModal);
        modal.addEventListener('click', (e) => {
            if (e.target === modal) closeModal();
        });

        // ESC para fechar
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && modal.classList.contains('show')) {
                closeModal();
            }
        });
    }

    // Abrir modal
    window.openModal = function(title, content, footer = '') {
        const modal = document.getElementById('cybernews-modal');
        const titleEl = modal.querySelector('.modal-title');
        const bodyEl = modal.querySelector('.modal-body');
        const footerEl = modal.querySelector('.modal-footer');

        titleEl.textContent = title;
        bodyEl.innerHTML = content;
        footerEl.innerHTML = footer;

        modal.classList.add('show');
        document.body.style.overflow = 'hidden';
    };

    // Fechar modal
    window.closeModal = function() {
        const modal = document.getElementById('cybernews-modal');
        modal.classList.remove('show');
        document.body.style.overflow = '';
    };

    // Modal de compra
    window.openPurchaseModal = function(productId, productName, price) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <h4 style="color: #90ee90; margin-bottom: 10px;">🛒 ${productName}</h4>
            <div style="font-size: 2rem; color: #90ee90; font-weight: bold;">${price}</div>
        </div>
        
        <div class="payment-options">
            <div class="payment-option" onclick="selectPayment('stripe')">
                <span class="payment-icon">💳</span>
                <div class="payment-name">Stripe</div>
                <small>Cartão de Crédito</small>
            </div>
            <div class="payment-option" onclick="selectPayment('pix')">
                <span class="payment-icon">🇧🇷</span>
                <div class="payment-name">PIX</div>
                <small>Pagamento Instantâneo</small>
            </div>
            <div class="payment-option" onclick="selectPayment('crypto')">
                <span class="payment-icon">₿</span>
                <div class="payment-name">Crypto</div>
                <small>Bitcoin/Ethereum</small>
            </div>
        </div>
        
        <div id="payment-details" style="margin-top: 20px; display: none;">
            <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 15px;">
                <p><strong>🔒 Pagamento Seguro</strong></p>
                <p>Processamento criptografado e dados protegidos.</p>
            </div>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            <i class="fas fa-times"></i> Cancelar
        </button>
        <button class="modal-btn primary" data-action="confirm-purchase" data-product-id="${productId}" id="confirm-purchase" disabled>
            <i class="fas fa-shopping-cart"></i> Confirmar Compra
        </button>`;

        openModal('💳 Finalizar Compra', content, footer);
        
        // Bind purchase button event
        setTimeout(() => {
            const modal = document.getElementById('cybernews-modal');
            if (modal) {
                modal.querySelector('[data-action="confirm-purchase"]')?.addEventListener('click', () => {
                    processPurchase(productId);
                });
            }
        }, 100);
    };

    // Selecionar método de pagamento
    window.selectPayment = function(method) {
        // Remover seleção anterior
        document.querySelectorAll('.payment-option').forEach(option => {
            option.classList.remove('selected');
        });

        // Selecionar novo
        event.target.closest('.payment-option').classList.add('selected');

        // Mostrar detalhes
        const details = document.getElementById('payment-details');
        const confirmBtn = document.getElementById('confirm-purchase');
        
        details.style.display = 'block';
        confirmBtn.disabled = false;
        confirmBtn.style.opacity = '1';

        // Atualizar detalhes baseado no método
        const methodDetails = {
            stripe: '💳 Redirecionando para Stripe Checkout...',
            pix: '🇧🇷 QR Code PIX será gerado após confirmação',
            crypto: '₿ Endereço de carteira será fornecido'
        };

        details.innerHTML = `
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 15px;">
            <p><strong>🔒 ${methodDetails[method]}</strong></p>
            <p>Processamento seguro e criptografado.</p>
        </div>`;
    };

    // Processar compra
    window.processPurchase = function(productId) {
        const selectedPayment = document.querySelector('.payment-option.selected');
        if (!selectedPayment) return;

        const paymentMethod = selectedPayment.onclick.toString().match(/'(.+?)'/)[1];
        
        // Simular processamento
        const content = `
        <div style="text-align: center;">
            <div style="font-size: 4rem; margin-bottom: 20px;">🔄</div>
            <h4 style="color: #90ee90;">Processando Pagamento...</h4>
            <p>Redirecionando para plataforma de pagamento segura.</p>
            <div style="width: 100%; height: 4px; background: rgba(144, 238, 144, 0.3); border-radius: 2px; overflow: hidden; margin: 20px 0;">
                <div style="width: 0%; height: 100%; background: #90ee90; animation: progress1 3s ease-out forwards;"></div>
            </div>
        </div>`;

        openModal('⚡ Processando...', content, '');

        // Simular redirecionamento após 3 segundos
        setTimeout(() => {
            closeModal();
            showSuccessMessage(productId, paymentMethod);
        }, 3000);
    };

    // Mensagem de sucesso
    function showSuccessMessage(productId, paymentMethod) {
        const content = `
        <div style="text-align: center;">
            <div style="font-size: 4rem; margin-bottom: 20px;">✅</div>
            <h4 style="color: #10b981;">Redirecionamento Realizado!</h4>
            <p>Você será redirecionado para a plataforma de pagamento em alguns segundos.</p>
            <div style="background: rgba(16, 185, 129, 0.1); border-radius: 10px; padding: 15px; margin: 20px 0;">
                <p><strong>📦 Produto:</strong> ${productId}</p>
                <p><strong>💳 Método:</strong> ${paymentMethod.toUpperCase()}</p>
                <p><strong>🔒 Status:</strong> Aguardando pagamento</p>
            </div>
        </div>`;

        const footer = `
        <button class="modal-btn primary" onclick="closeModal()">
            <i class="fas fa-check"></i> Entendi
        </button>`;

        openModal('🎉 Quase lá!', content, footer);
    }

    // Modal para VPNs
    window.openVPNModal = function(vpnName, promoCode, originalPrice, promoPrice) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <h4 style="color: #90ee90; margin-bottom: 15px;">🌿 ${vpnName}</h4>
            <div style="display: flex; align-items: center; justify-content: center; gap: 15px; margin: 20px 0;">
                <span style="text-decoration: line-through; color: rgba(255,255,255,0.6);">${originalPrice}</span>
                <span style="font-size: 2rem; color: #90ee90; font-weight: bold;">${promoPrice}</span>
            </div>
        </div>
        
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 20px; margin: 20px 0;">
            <h5 style="color: #90ee90; margin-bottom: 10px;">🎁 Código Promocional:</h5>
            <div style="background: rgba(45, 80, 22, 0.8); padding: 15px; border-radius: 10px; text-align: center; cursor: pointer;" data-action="copy-promo" data-promo="${promoCode}">
                <code style="color: #90ee90; font-size: 1.2rem; font-weight: bold;">${promoCode}</code>
                <p style="margin: 5px 0 0 0; font-size: 0.8rem; color: rgba(255,255,255,0.7);">Clique para copiar</p>
            </div>
        </div>

        <div style="background: rgba(16, 185, 129, 0.1); border-radius: 10px; padding: 15px;">
            <p><strong>🛡️ Garantia:</strong> 30 dias ou seu dinheiro de volta</p>
            <p><strong>⚡ Ativação:</strong> Instantânea após pagamento</p>
            <p><strong>🌍 Servidores:</strong> Globais com baixa latência</p>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            <i class="fas fa-times"></i> Fechar
        </button>
        <button class="modal-btn primary" onclick="window.open('#', '_blank')">
            <i class="fas fa-rocket"></i> Ativar ${vpnName}
        </button>`;

        openModal(`🌿 ${vpnName} Premium`, content, footer);
        
        // Bind VPN events
        setTimeout(() => {
            const modal = document.getElementById('cybernews-modal');
            if (modal) {
                modal.querySelector('[data-action="copy-promo"]')?.addEventListener('click', (e) => {
                    const promo = e.target.closest('[data-promo]')?.dataset.promo;
                    if (promo) copyToClipboard(promo);
                });
            }
        }, 100);
    };

    // Modal de análise de email
    window.openEmailAnalysis = function() {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">📧</div>
            <p>Cole o cabeçalho do email suspeito abaixo para análise:</p>
        </div>
        
        <textarea id="email-headers" placeholder="Cole aqui os headers do email..." style="width: 100%; height: 150px; background: rgba(0,0,0,0.5); border: 2px solid rgba(144,238,144,0.3); border-radius: 10px; padding: 15px; color: #fff; font-family: monospace; resize: vertical;"></textarea>
        
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 15px; margin-top: 15px;">
            <p><strong>🔍 Nossa análise inclui:</strong></p>
            <ul style="margin: 10px 0; padding-left: 20px;">
                <li>Verificação de domínio e SPF</li>
                <li>Análise de links maliciosos</li>
                <li>Detecção de phishing</li>
                <li>Score de confiabilidade</li>
            </ul>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            <i class="fas fa-times"></i> Cancelar
        </button>
        <button class="modal-btn primary" onclick="analyzeEmail()">
            <i class="fas fa-search"></i> Analisar Email
        </button>`;

        openModal('📧 Análise de Email Suspeito', content, footer);
    };

    // Analisar email
    window.analyzeEmail = function() {
        const headers = document.getElementById('email-headers').value;
        if (!headers.trim()) {
            alert('Por favor, cole os headers do email.');
            return;
        }

        const content = `
        <div style="text-align: center;">
            <div style="font-size: 4rem; margin-bottom: 20px;">🔍</div>
            <h4 style="color: #90ee90;">Analisando Email...</h4>
            <div style="width: 100%; height: 4px; background: rgba(144, 238, 144, 0.3); border-radius: 2px; overflow: hidden; margin: 20px 0;">
                <div style="width: 0%; height: 100%; background: #90ee90; animation: progress1 3s ease-out forwards;"></div>
            </div>
        </div>`;

        openModal('🔬 Análise em Progresso...', content, '');

        // Simular análise
        setTimeout(() => {
            showEmailResults();
        }, 3000);
    };

    // Modal para contratação de serviços
    window.openServiceModal = function(serviceId, serviceName, price) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">🤝</div>
            <h3 style="color: #90ee90; margin-bottom: 10px;">${serviceName}</h3>
            <p style="color: #4a7c23; font-size: 1.2rem; font-weight: bold;">${price}</p>
        </div>
        
        <form id="serviceForm" style="text-align: left;">
            <div style="margin-bottom: 20px;">
                <label style="display: block; margin-bottom: 8px; color: #90ee90; font-weight: bold;">Nome da Empresa:</label>
                <input type="text" name="company" required placeholder="Sua empresa" 
                       style="width: 100%; padding: 12px; background: rgba(0,0,0,0.5); border: 2px solid rgba(144,238,144,0.3); border-radius: 8px; color: #fff;">
            </div>
            
            <div style="margin-bottom: 20px;">
                <label style="display: block; margin-bottom: 8px; color: #90ee90; font-weight: bold;">Email de Contato:</label>
                <input type="email" name="email" required placeholder="contato@empresa.com"
                       style="width: 100%; padding: 12px; background: rgba(0,0,0,0.5); border: 2px solid rgba(144,238,144,0.3); border-radius: 8px; color: #fff;">
            </div>
            
            <div style="margin-bottom: 20px;">
                <label style="display: block; margin-bottom: 8px; color: #90ee90; font-weight: bold;">Telefone:</label>
                <input type="tel" name="phone" required placeholder="(11) 99999-9999"
                       style="width: 100%; padding: 12px; background: rgba(0,0,0,0.5); border: 2px solid rgba(144,238,144,0.3); border-radius: 8px; color: #fff;">
            </div>
            
            <div style="margin-bottom: 20px;">
                <label style="display: block; margin-bottom: 8px; color: #90ee90; font-weight: bold;">Tamanho da Empresa:</label>
                <select name="company_size" required style="width: 100%; padding: 12px; background: rgba(0,0,0,0.5); border: 2px solid rgba(144,238,144,0.3); border-radius: 8px; color: #fff;">
                    <option value="">Selecione...</option>
                    <option value="startup">Startup (1-10 funcionários)</option>
                    <option value="small">Pequena (11-50 funcionários)</option>
                    <option value="medium">Média (51-200 funcionários)</option>
                    <option value="large">Grande (200+ funcionários)</option>
                </select>
            </div>
            
            <div style="margin-bottom: 20px;">
                <label style="display: block; margin-bottom: 8px; color: #90ee90; font-weight: bold;">Detalhes do Projeto:</label>
                <textarea name="details" rows="4" placeholder="Descreva suas necessidades específicas..."
                          style="width: 100%; padding: 12px; background: rgba(0,0,0,0.5); border: 2px solid rgba(144,238,144,0.3); border-radius: 8px; color: #fff; resize: vertical;"></textarea>
            </div>
            
            <div style="margin-bottom: 20px;">
                <label style="display: block; margin-bottom: 8px; color: #90ee90; font-weight: bold;">Prazo Desejado:</label>
                <select name="timeline" required style="width: 100%; padding: 12px; background: rgba(0,0,0,0.5); border: 2px solid rgba(144,238,144,0.3); border-radius: 8px; color: #fff;">
                    <option value="">Selecione...</option>
                    <option value="urgent">Urgente (1-2 semanas)</option>
                    <option value="normal">Normal (1 mês)</option>
                    <option value="flexible">Flexível (2+ meses)</option>
                </select>
            </div>
        </form>`;

        const footer = `
        <button class="modal-btn secondary" data-action="whatsapp" data-service-id="${serviceId}" data-service-name="${serviceName}">
            <i class="fab fa-whatsapp"></i> WhatsApp
        </button>
        <button class="modal-btn primary" data-action="submit-service" data-service-id="${serviceId}" data-service-name="${serviceName}">
            <i class="fas fa-paper-plane"></i> Enviar Solicitação
        </button>`;

        openModal('🤝 Contratação de Serviço', content, footer);
        
        // Bind events after modal is created
        setTimeout(() => {
            const modal = document.getElementById('cybernews-modal');
            if (modal) {
                modal.querySelector('[data-action="whatsapp"]')?.addEventListener('click', () => {
                    contactWhatsApp(serviceId, serviceName);
                });
                modal.querySelector('[data-action="submit-service"]')?.addEventListener('click', () => {
                    submitServiceForm(serviceId, serviceName);
                });
            }
        }, 100);
    };

    // Submeter formulário de serviço
    window.submitServiceForm = function(serviceId, serviceName) {
        const form = document.getElementById('serviceForm');
        const formData = new FormData(form);
        
        if (!form.checkValidity()) {
            alert('Por favor, preencha todos os campos obrigatórios.');
            return;
        }

        // Simular envio
        const content = `
        <div style="text-align: center;">
            <div style="font-size: 4rem; margin-bottom: 20px;">✅</div>
            <h3 style="color: #90ee90;">Solicitação Enviada!</h3>
            <p>Recebemos sua solicitação para <strong>${serviceName}</strong></p>
            <p>Nossa equipe entrará em contato em até 24 horas.</p>
            
            <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 20px; margin: 20px 0;">
                <h4>📧 Próximos Passos:</h4>
                <ul style="text-align: left; margin: 10px 0;">
                    <li>Análise detalhada da sua necessidade</li>
                    <li>Proposta comercial personalizada</li>
                    <li>Agendamento de reunião técnica</li>
                    <li>Plano de implementação</li>
                </ul>
            </div>
        </div>`;

        const footer = `
        <button class="modal-btn primary" onclick="closeModal()">
            <i class="fas fa-check"></i> Entendi
        </button>`;

        openModal('✅ Solicitação Enviada', content, footer);
    };

    // Contato via WhatsApp
    window.contactWhatsApp = function(serviceId, serviceName) {
        const message = `Olá! Tenho interesse no serviço: ${serviceName}. Gostaria de mais informações.`;
        const phone = '5511999999999'; // Substitua pelo número real
        const url = `https://wa.me/${phone}?text=${encodeURIComponent(message)}`;
        window.open(url, '_blank');
    };

    // Resultados da análise
    function showEmailResults() {
        const content = `
        <div style="text-align: center; margin-bottom: 20px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">⚠️</div>
            <h4 style="color: #ff6b35;">Email Suspeito Detectado!</h4>
        </div>
        
        <div style="background: rgba(255, 107, 53, 0.1); border: 1px solid rgba(255, 107, 53, 0.3); border-radius: 10px; padding: 20px;">
            <h5 style="color: #ff6b35; margin-bottom: 15px;">🚨 Ameaças Encontradas:</h5>
            <ul>
                <li>✅ Domínio suspeito detectado</li>
                <li>✅ Link de phishing identificado</li>
                <li>⚠️ SPF/DKIM falharam</li>
                <li>🔍 Padrões de spam encontrados</li>
            </ul>
        </div>

        <div style="background: rgba(16, 185, 129, 0.1); border-radius: 10px; padding: 15px; margin-top: 15px;">
            <h5 style="color: #10b981;">🛡️ Recomendações:</h5>
            <ul>
                <li>❌ NÃO clique em nenhum link</li>
                <li>🗑️ Delete o email imediatamente</li>
                <li>📢 Reporte para sua equipe de TI</li>
                <li>🔒 Mude senhas se já clicou</li>
            </ul>
        </div>`;

        const footer = `
        <button class="modal-btn primary" onclick="closeModal()">
            <i class="fas fa-shield-alt"></i> Entendi, Obrigado!
        </button>`;

        openModal('🛡️ Análise Concluída', content, footer);
    }

    // Copiar para clipboard
    window.copyToClipboard = function(text) {
        navigator.clipboard.writeText(text).then(() => {
            // Feedback visual
            const element = event.target.closest('[onclick]');
            const original = element.innerHTML;
            element.innerHTML = '<strong style="color: #10b981;">✅ COPIADO!</strong>';
            setTimeout(() => {
                element.innerHTML = original;
            }, 2000);
        });
    };

    // Inicializar quando DOM estiver pronto
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', createModalStructure);
    } else {
        createModalStructure();
    }

    // Bind automático para botões de compra e serviços (EXCETO na página da loja)
    document.addEventListener('click', (e) => {
        // Se estamos na página da loja, deixar o store.js lidar com os botões
        if (window.location.pathname === '/loja' || window.location.pathname.includes('loja.php')) {
            return; // Não interceptar na loja
        }
        
        const buyBtn = e.target.closest('.buy-button, .service-btn, button[data-product-id]');
        if (buyBtn) {
            e.preventDefault();
            e.stopPropagation();
            
            const productId = buyBtn.dataset.productId || buyBtn.getAttribute('data-product-id') || 'produto-generico';
            const productCard = buyBtn.closest('.product-card, .service-card, .card');
            const productName = productCard?.querySelector('h3, .card-title')?.textContent || buyBtn.getAttribute('aria-label') || 'Produto';
            const price = productCard?.querySelector('.product-price, .service-price, .price')?.textContent || 'R$ 99,00';
            
            // Verificar se é serviço (botão tem classe service-btn ou texto "Contratar")
            const isService = buyBtn.classList.contains('service-btn') || 
                            buyBtn.textContent.toLowerCase().includes('contratar') ||
                            productId.includes('setup') || productId.includes('config') || 
                            productId.includes('training') || productId.includes('migration');
            
            if (isService) {
                openServiceModal(productId, productName, price);
            } else {
                openPurchaseModal(productId, productName, price);
            }
            return false;
        }

        // VPN buttons - redirecionar para site oficial
        const vpnBtn = e.target.closest('.vpn-cta-btn');
        if (vpnBtn) {
            e.preventDefault();
            e.stopPropagation();
            const vpnCard = vpnBtn.closest('.vpn-card');
            const vpnName = vpnCard?.querySelector('h3')?.textContent || 'VPN';
            const promoCode = vpnCard?.querySelector('.promo-code span')?.textContent || 'PROMO';
            
            // Abrir modal de redirecionamento para VPN
            openVPNRedirectModal(vpnName, promoCode);
            return false;
        }

        // Email analysis
        if (e.target.closest('[onclick*="analyzeEmail"]')) {
            e.preventDefault();
            openEmailAnalysis();
        }

        // Botões genéricos com href="#" - adicionar funcionalidades (exceto navegação)
        const genericBtn = e.target.closest('a[href="#"], button[onclick=""]');
        if (genericBtn && !genericBtn.closest('.modal-overlay')) {
            // Verificar se não é um link de navegação (menu, social, etc)
            const isNavigationLink = genericBtn.closest('.menu-items, .social-icons, .back-to-top, .menu-toggle');
            const hasInteractiveClass = 
                genericBtn.classList.contains('guide-btn') ||
                genericBtn.classList.contains('tool-link') ||
                genericBtn.classList.contains('framework-btn') ||
                genericBtn.classList.contains('analysis-btn') ||
                genericBtn.classList.contains('vpn-cta-btn') ||
                genericBtn.classList.contains('training-btn') ||
                genericBtn.classList.contains('compliance-btn') ||
                genericBtn.classList.contains('family-btn') ||
                genericBtn.classList.contains('education-btn');
            
            const hasInteractiveText = 
                genericBtn.textContent.toLowerCase().includes('download') ||
                genericBtn.textContent.toLowerCase().includes('experimentar') ||
                genericBtn.textContent.toLowerCase().includes('ler guia') ||
                genericBtn.textContent.toLowerCase().includes('ver detalhes') ||
                genericBtn.textContent.toLowerCase().includes('analisar') ||
                genericBtn.textContent.toLowerCase().includes('configurar') ||
                genericBtn.textContent.toLowerCase().includes('implementar') ||
                genericBtn.textContent.toLowerCase().includes('treinar') ||
                genericBtn.textContent.toLowerCase().includes('estudar') ||
                genericBtn.textContent.toLowerCase().includes('ativar') ||
                genericBtn.textContent.toLowerCase().includes('usar') ||
                genericBtn.textContent.toLowerCase().includes('avaliar');
            
            if (!isNavigationLink && (hasInteractiveClass || hasInteractiveText)) {
                console.log('🎯 Botão detectado:', genericBtn.textContent, 'Classes:', genericBtn.className);
                e.preventDefault();
                e.stopPropagation();
                handleGenericButton(genericBtn);
                return false;
            } else if (genericBtn) {
                console.log('❌ Botão ignorado:', genericBtn.textContent, 'IsNav:', !!isNavigationLink, 'HasClass:', hasInteractiveClass, 'HasText:', hasInteractiveText);
                console.log('🔍 Detalhes:', {
                    text: genericBtn.textContent.toLowerCase(),
                    classes: genericBtn.className,
                    href: genericBtn.href
                });
            }
        }
    });

    // Função para lidar com botões genéricos
    function handleGenericButton(btn) {
        console.log('🎯 handleGenericButton CHAMADA!', btn.textContent, btn.className);
        
        const text = btn.textContent.toLowerCase();
        const classes = btn.className;
        const parentSection = btn.closest('section, .card, article')?.querySelector('h3, h4, .card-title')?.textContent || 'Recurso';
        
        // Identificar tipo de ação baseado no texto/classe
        if (text.includes('download') || text.includes('baixar')) {
            showDownloadModal(parentSection, btn);
        } else if (text.includes('demo') || text.includes('experimentar') || text.includes('testar')) {
            showDemoModal(parentSection, btn);
        } else if (text.includes('configurar') || text.includes('setup') || text.includes('implementar')) {
            showConfigModal(parentSection, btn);
        } else if (text.includes('ler') || text.includes('guia') || text.includes('estudar')) {
            console.log('📚 Detectado: Guia - Chamando showGuideModal');
            showGuideModal(parentSection, btn);
        } else if (text.includes('certificação') || text.includes('treinar') || text.includes('programa')) {
            showTrainingModal(parentSection, btn);
        } else if (text.includes('analisar') || text.includes('avaliar') || text.includes('relatório')) {
            showAnalysisModal(parentSection, btn);
        } else if (classes.includes('social-icon')) {
            showSocialModal(btn);
        } else {
            showGenericInfoModal(parentSection, btn);
        }
    }

    // Modal de download inovador
    function showDownloadModal(resource, btn) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 4rem; margin-bottom: 15px; animation: bounce 2s infinite;">📥</div>
            <h3 style="color: #90ee90; text-shadow: 0 0 15px rgba(144, 238, 144, 0.6);">Download: ${resource}</h3>
        </div>
        
        <!-- Simulador de download -->
        <div style="background: linear-gradient(135deg, rgba(144, 238, 144, 0.05), rgba(144, 238, 144, 0.15)); border-radius: 15px; padding: 25px; margin: 20px 0; border: 2px solid rgba(144, 238, 144, 0.3);">
            <h4 style="color: #90ee90; margin-bottom: 15px;">🚀 Download Simulado</h4>
            <div style="background: rgba(0,0,0,0.3); border-radius: 10px; padding: 15px; margin: 15px 0;">
                <div style="display: flex; justify-content: space-between; margin-bottom: 10px;">
                    <span style="color: #b9f9b9;">📦 ${resource}</span>
                    <span style="color: #90ee90;">25.6 MB</span>
                </div>
                <div style="width: 100%; height: 8px; background: rgba(144, 238, 144, 0.2); border-radius: 4px; overflow: hidden;">
                    <div style="width: 0%; height: 100%; background: linear-gradient(90deg, #90ee90, #b9f9b9); border-radius: 4px; animation: downloadProgress 3s ease-out forwards;"></div>
                </div>
                <div style="margin-top: 10px; color: #b9f9b9; font-size: 0.9rem;">
                    <span id="download-status">Preparando download...</span>
                </div>
            </div>
            
            <div style="background: rgba(16, 185, 129, 0.1); border-radius: 10px; padding: 15px; margin: 15px 0;">
                <h5 style="color: #10b981; margin-bottom: 10px;">✨ Recursos Incluídos:</h5>
                <ul style="text-align: left; margin: 10px 0; color: #b9f9b9;">
                    <li>• Código fonte completo</li>
                    <li>• Documentação detalhada</li>
                    <li>• Exemplos de uso</li>
                    <li>• Scripts de instalação</li>
                    <li>• Suporte técnico por 30 dias</li>
                </ul>
            </div>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            ❌ Cancelar
        </button>
        <button class="modal-btn primary" onclick="startFakeDownload('${resource}')">
            🚀 Iniciar Download
        </button>`;

        openModal('📥 Centro de Downloads', content, footer);
    }

    // Modal de demo/teste
    function showDemoModal(resource, btn) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">🚀</div>
            <h3 style="color: #90ee90;">Demo: ${resource}</h3>
        </div>
        
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 20px; margin: 20px 0;">
            <h4>🎯 O que você pode testar:</h4>
            <ul style="text-align: left; margin: 10px 0;">
                <li>Interface completa da ferramenta</li>
                <li>Funcionalidades principais</li>
                <li>Relatórios de exemplo</li>
                <li>Configurações básicas</li>
            </ul>
        </div>
        
        <div style="background: rgba(255, 215, 0, 0.1); border-radius: 10px; padding: 15px; margin: 15px 0;">
            <p><strong>🔧 Em Desenvolvimento:</strong> Demo interativo chegando em breve!</p>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            <i class="fas fa-times"></i> Fechar
        </button>
        <button class="modal-btn primary" onclick="requestDemo('${resource}')">
            <i class="fas fa-calendar"></i> Agendar Demo
        </button>`;

        openModal('🚀 Demo Interativo', content, footer);
    }

    // Modal de configuração
    function showConfigModal(resource, btn) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">⚙️</div>
            <h3 style="color: #90ee90;">Configuração: ${resource}</h3>
        </div>
        
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 20px; margin: 20px 0;">
            <h4>🛠️ Processo de Setup:</h4>
            <ol style="text-align: left; margin: 10px 0;">
                <li>Análise do ambiente atual</li>
                <li>Planejamento da implementação</li>
                <li>Configuração personalizada</li>
                <li>Testes e validação</li>
                <li>Documentação e treinamento</li>
            </ol>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            <i class="fas fa-times"></i> Fechar
        </button>
        <button class="modal-btn primary" onclick="openServiceModal('config-${resource}', 'Configuração ${resource}', 'Sob consulta')">
            <i class="fas fa-cogs"></i> Solicitar Configuração
        </button>`;

        openModal('⚙️ Configuração Personalizada', content, footer);
    }

    // Modal de guia no estilo Grok futurista
    function showGuideModal(resource, btn) {
        console.log('🧠 showGuideModal CHAMADA!', resource, btn);
        
        // Detectar tipo de guia baseado no texto
        const guideType = btn.textContent.toLowerCase();
        let guideContent = '';
        let guideIcon = '🧠';
        let guideTitle = 'Guia Inteligente';
        
        if (guideType.includes('navegação') || guideType.includes('anônima')) {
            guideIcon = '🕵️';
            guideTitle = 'Navegação Anônima';
            guideContent = `
            <div class="grok-section">
                <div class="grok-header">
                    <span class="grok-icon">🔧</span>
                    <h4>Configuração Tor Browser</h4>
                </div>
                <div class="grok-content">
                    <div class="grok-step">
                        <span class="step-number">01</span>
                        <div class="step-content">
                            <h5>Download Oficial</h5>
                            <p>Baixe apenas do site oficial torproject.org</p>
                        </div>
                    </div>
                    <div class="grok-step">
                        <span class="step-number">02</span>
                        <div class="step-content">
                            <h5>Nível de Segurança</h5>
                            <p>Configure para "Mais Seguro" nas configurações</p>
                        </div>
                    </div>
                    <div class="grok-step">
                        <span class="step-number">03</span>
                        <div class="step-content">
                            <h5>JavaScript</h5>
                            <p>Desative quando possível para máxima privacidade</p>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="grok-section">
                <div class="grok-header">
                    <span class="grok-icon">🔗</span>
                    <h4>VPN + Tor (Double VPN)</h4>
                </div>
                <div class="grok-content">
                    <div class="grok-warning">
                        <span class="warning-icon">⚠️</span>
                        <p>Conecte primeiro à VPN, depois abra o Tor Browser</p>
                    </div>
                    <div class="grok-tip">
                        <span class="tip-icon">💡</span>
                        <p>Verifique seu IP em whatismyipaddress.com</p>
                    </div>
                </div>
            </div>`;
        } else if (guideType.includes('mobile') || guideType.includes('móvel')) {
            guideIcon = '📱';
            guideTitle = 'Privacidade Mobile';
            guideContent = `
            <div class="grok-section">
                <div class="grok-header">
                    <span class="grok-icon">🍎</span>
                    <h4>iOS (iPhone/iPad)</h4>
                </div>
                <div class="grok-content">
                    <div class="grok-grid">
                        <div class="grok-card">
                            <h5>Rastreamento</h5>
                            <p>Ajustes > Privacidade > Rastreamento > Desativar</p>
                        </div>
                        <div class="grok-card">
                            <h5>Safari</h5>
                            <p>Ajustes > Safari > Prevenir rastreamento</p>
                        </div>
                        <div class="grok-card">
                            <h5>Private Relay</h5>
                            <p>Ativar iCloud Private Relay</p>
                        </div>
                        <div class="grok-card">
                            <h5>Apps Seguros</h5>
                            <p>Signal, ProtonMail, Element</p>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="grok-section">
                <div class="grok-header">
                    <span class="grok-icon">🤖</span>
                    <h4>Android</h4>
                </div>
                <div class="grok-content">
                    <div class="grok-grid">
                        <div class="grok-card">
                            <h5>Ads ID</h5>
                            <p>Configurações > Privacidade > Desativar ads ID</p>
                        </div>
                        <div class="grok-card">
                            <h5>Navegador</h5>
                            <p>Firefox com uBlock Origin</p>
                        </div>
                        <div class="grok-card">
                            <h5>VPN</h5>
                            <p>Wireguard sempre ativa</p>
                        </div>
                        <div class="grok-card">
                            <h5>Apps</h5>
                            <p>Signal, Element, F-Droid</p>
                        </div>
                    </div>
                </div>
            </div>`;
        } else if (guideType.includes('email')) {
            guideIcon = '📧';
            guideTitle = 'Email Seguro';
            guideContent = `
            <div class="grok-section">
                <div class="grok-header">
                    <span class="grok-icon">🔐</span>
                    <h4>ProtonMail vs Tutanota</h4>
                </div>
                <div class="grok-content">
                    <div class="grok-comparison">
                        <div class="grok-option">
                            <div class="option-header">
                                <span class="option-icon">🇨🇭</span>
                                <h5>ProtonMail</h5>
                            </div>
                            <div class="option-features">
                                <div class="feature">✓ Baseado na Suíça</div>
                                <div class="feature">✓ Zero-knowledge</div>
                                <div class="feature">✓ PGP integrado</div>
                                <div class="feature">✓ Calendar & Drive</div>
                            </div>
                        </div>
                        <div class="grok-option">
                            <div class="option-header">
                                <span class="option-icon">🇩🇪</span>
                                <h5>Tutanota</h5>
                            </div>
                            <div class="option-features">
                                <div class="feature">✓ Baseado na Alemanha</div>
                                <div class="feature">✓ Código aberto</div>
                                <div class="feature">✓ Criptografia quântica</div>
                                <div class="feature">✓ Calendar integrado</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>`;
        } else if (guideType.includes('crypto') || guideType.includes('moeda')) {
            guideIcon = '💰';
            guideTitle = 'Criptomoedas Privadas';
            guideContent = `
            <div class="grok-section">
                <div class="grok-header">
                    <span class="grok-icon">🪙</span>
                    <h4>Moedas de Privacidade</h4>
                </div>
                <div class="grok-content">
                    <div class="grok-crypto">
                        <div class="crypto-card">
                            <div class="crypto-header">
                                <span class="crypto-icon">🔒</span>
                                <h5>Monero (XMR)</h5>
                            </div>
                            <div class="crypto-details">
                                <p>Privacidade total por padrão</p>
                                <div class="crypto-features">
                                    <span class="feature-tag">Ring Signatures</span>
                                    <span class="feature-tag">Stealth Addresses</span>
                                    <span class="feature-tag">RingCT</span>
                                </div>
                            </div>
                        </div>
                        <div class="crypto-card">
                            <div class="crypto-header">
                                <span class="crypto-icon">👁️</span>
                                <h5>Zcash (ZEC)</h5>
                            </div>
                            <div class="crypto-details">
                                <p>Transações blindadas opcionais</p>
                                <div class="crypto-features">
                                    <span class="feature-tag">zk-SNARKs</span>
                                    <span class="feature-tag">Shielded Pools</span>
                                    <span class="feature-tag">Selective Disclosure</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>`;
        } else {
            // Guia genérico
            guideContent = `
            <div class="grok-section">
                <div class="grok-header">
                    <span class="grok-icon">📚</span>
                    <h4>${resource}</h4>
                </div>
                <div class="grok-content">
                    <div class="grok-outline">
                        <div class="outline-item">
                            <span class="outline-icon">🎯</span>
                            <h5>Conceitos Fundamentais</h5>
                        </div>
                        <div class="outline-item">
                            <span class="outline-icon">⚙️</span>
                            <h5>Implementação Prática</h5>
                        </div>
                        <div class="outline-item">
                            <span class="outline-icon">⭐</span>
                            <h5>Melhores Práticas</h5>
                        </div>
                        <div class="outline-item">
                            <span class="outline-icon">🔧</span>
                            <h5>Troubleshooting</h5>
                        </div>
                        <div class="outline-item">
                            <span class="outline-icon">💻</span>
                            <h5>Exemplos de Código</h5>
                        </div>
                    </div>
                </div>
            </div>`;
        }

        // Criar modal no estilo Grok
        const modal = document.createElement('div');
        modal.className = 'modal-overlay grok-modal';
        modal.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.95);
            backdrop-filter: blur(20px);
            z-index: 99999;
            display: flex;
            align-items: center;
            justify-content: center;
            animation: grokFadeIn 0.5s ease-out;
        `;

        modal.innerHTML = `
            <div class="grok-modal-content" style="
                background: linear-gradient(135deg, 
                    rgba(0, 0, 0, 0.9) 0%, 
                    rgba(20, 20, 40, 0.95) 50%, 
                    rgba(0, 0, 0, 0.9) 100%);
                border: 2px solid rgba(144, 238, 144, 0.3);
                border-radius: 20px;
                padding: 0;
                max-width: 800px;
                width: 90%;
                max-height: 90vh;
                overflow: hidden;
                position: relative;
                box-shadow: 
                    0 0 50px rgba(144, 238, 144, 0.3),
                    inset 0 0 50px rgba(144, 238, 144, 0.1);
                animation: grokSlideIn 0.6s ease-out;
            ">
                <!-- Grok Header -->
                <div class="grok-modal-header" style="
                    background: linear-gradient(135deg, rgba(144, 238, 144, 0.1), rgba(144, 238, 144, 0.2));
                    border-bottom: 1px solid rgba(144, 238, 144, 0.3);
                    padding: 25px 30px;
                    border-radius: 18px 18px 0 0;
                    position: relative;
                    overflow: hidden;
                ">
                    <div class="grok-header-bg" style="
                        position: absolute;
                        top: 0;
                        left: 0;
                        right: 0;
                        bottom: 0;
                        background: 
                            radial-gradient(circle at 20% 20%, rgba(144, 238, 144, 0.1) 0%, transparent 50%),
                            radial-gradient(circle at 80% 80%, rgba(144, 238, 144, 0.05) 0%, transparent 50%);
                        animation: grokFloat 6s ease-in-out infinite;
                    "></div>
                    <div style="position: relative; z-index: 2; display: flex; align-items: center; gap: 20px;">
                        <div class="grok-icon-large" style="
                            font-size: 3rem;
                            animation: grokPulse 3s ease-in-out infinite;
                            filter: drop-shadow(0 0 20px rgba(144, 238, 144, 0.6));
                        ">${guideIcon}</div>
                        <div>
                            <h2 style="
                                color: #90ee90;
                                margin: 0;
                                font-size: 2rem;
                                text-shadow: 0 0 20px rgba(144, 238, 144, 0.6);
                                font-weight: 700;
                            ">${guideTitle}</h2>
                            <p style="
                                color: #b9f9b9;
                                margin: 5px 0 0 0;
                                font-size: 1rem;
                                opacity: 0.8;
                            ">Guia Inteligente • ${resource}</p>
                        </div>
                    </div>
                    <button onclick="closeModal()" style="
                        position: absolute;
                        top: 20px;
                        right: 20px;
                        background: rgba(255, 255, 255, 0.1);
                        border: 1px solid rgba(255, 255, 255, 0.2);
                        color: #90ee90;
                        width: 40px;
                        height: 40px;
                        border-radius: 50%;
                        cursor: pointer;
                        font-size: 1.2rem;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        transition: all 0.3s ease;
                        backdrop-filter: blur(10px);
                    " onmouseover="this.style.background='rgba(255, 255, 255, 0.2)'; this.style.transform='scale(1.1)'" 
                       onmouseout="this.style.background='rgba(255, 255, 255, 0.1)'; this.style.transform='scale(1)'">
                        ✕
                    </button>
                </div>

                <!-- Grok Content -->
                <div class="grok-modal-body" style="
                    padding: 30px;
                    max-height: 60vh;
                    overflow-y: auto;
                    background: linear-gradient(135deg, rgba(0, 0, 0, 0.3), rgba(20, 20, 40, 0.2));
                ">
                    <style>
                        .grok-section {
                            margin-bottom: 30px;
                            background: rgba(0, 0, 0, 0.3);
                            border-radius: 15px;
                            padding: 25px;
                            border: 1px solid rgba(144, 238, 144, 0.2);
                            position: relative;
                            overflow: hidden;
                        }
                        .grok-section::before {
                            content: '';
                            position: absolute;
                            top: 0;
                            left: 0;
                            right: 0;
                            height: 2px;
                            background: linear-gradient(90deg, transparent, rgba(144, 238, 144, 0.5), transparent);
                            animation: grokScan 3s linear infinite;
                        }
                        .grok-header {
                            display: flex;
                            align-items: center;
                            gap: 15px;
                            margin-bottom: 20px;
                        }
                        .grok-icon {
                            font-size: 1.5rem;
                            filter: drop-shadow(0 0 10px rgba(144, 238, 144, 0.6));
                        }
                        .grok-header h4 {
                            color: #90ee90;
                            margin: 0;
                            font-size: 1.3rem;
                            text-shadow: 0 0 10px rgba(144, 238, 144, 0.4);
                        }
                        .grok-step {
                            display: flex;
                            align-items: flex-start;
                            gap: 15px;
                            margin-bottom: 15px;
                            padding: 15px;
                            background: rgba(144, 238, 144, 0.05);
                            border-radius: 10px;
                            border-left: 3px solid rgba(144, 238, 144, 0.5);
                        }
                        .step-number {
                            background: linear-gradient(135deg, rgba(144, 238, 144, 0.2), rgba(144, 238, 144, 0.3));
                            color: #90ee90;
                            width: 30px;
                            height: 30px;
                            border-radius: 50%;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                            font-weight: bold;
                            font-size: 0.9rem;
                            flex-shrink: 0;
                        }
                        .step-content h5 {
                            color: #90ee90;
                            margin: 0 0 5px 0;
                            font-size: 1rem;
                        }
                        .step-content p {
                            color: #b9f9b9;
                            margin: 0;
                            font-size: 0.9rem;
                            line-height: 1.4;
                        }
                        .grok-warning, .grok-tip {
                            display: flex;
                            align-items: center;
                            gap: 10px;
                            padding: 15px;
                            border-radius: 10px;
                            margin: 10px 0;
                        }
                        .grok-warning {
                            background: rgba(255, 193, 7, 0.1);
                            border: 1px solid rgba(255, 193, 7, 0.3);
                        }
                        .grok-tip {
                            background: rgba(16, 185, 129, 0.1);
                            border: 1px solid rgba(16, 185, 129, 0.3);
                        }
                        .warning-icon, .tip-icon {
                            font-size: 1.2rem;
                        }
                        .grok-warning p, .grok-tip p {
                            color: #b9f9b9;
                            margin: 0;
                            font-size: 0.9rem;
                        }
                        .grok-grid {
                            display: grid;
                            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                            gap: 15px;
                        }
                        .grok-card {
                            background: rgba(144, 238, 144, 0.05);
                            border: 1px solid rgba(144, 238, 144, 0.2);
                            border-radius: 10px;
                            padding: 15px;
                            transition: all 0.3s ease;
                        }
                        .grok-card:hover {
                            background: rgba(144, 238, 144, 0.1);
                            transform: translateY(-2px);
                        }
                        .grok-card h5 {
                            color: #90ee90;
                            margin: 0 0 8px 0;
                            font-size: 1rem;
                        }
                        .grok-card p {
                            color: #b9f9b9;
                            margin: 0;
                            font-size: 0.85rem;
                            line-height: 1.4;
                        }
                        .grok-comparison {
                            display: grid;
                            grid-template-columns: 1fr 1fr;
                            gap: 20px;
                        }
                        .grok-option {
                            background: rgba(144, 238, 144, 0.05);
                            border: 1px solid rgba(144, 238, 144, 0.2);
                            border-radius: 15px;
                            padding: 20px;
                        }
                        .option-header {
                            display: flex;
                            align-items: center;
                            gap: 10px;
                            margin-bottom: 15px;
                        }
                        .option-icon {
                            font-size: 1.5rem;
                        }
                        .option-header h5 {
                            color: #90ee90;
                            margin: 0;
                            font-size: 1.2rem;
                        }
                        .feature {
                            color: #b9f9b9;
                            margin: 8px 0;
                            font-size: 0.9rem;
                        }
                        .grok-crypto {
                            display: grid;
                            grid-template-columns: 1fr 1fr;
                            gap: 20px;
                        }
                        .crypto-card {
                            background: rgba(144, 238, 144, 0.05);
                            border: 1px solid rgba(144, 238, 144, 0.2);
                            border-radius: 15px;
                            padding: 20px;
                        }
                        .crypto-header {
                            display: flex;
                            align-items: center;
                            gap: 10px;
                            margin-bottom: 15px;
                        }
                        .crypto-icon {
                            font-size: 1.5rem;
                        }
                        .crypto-header h5 {
                            color: #90ee90;
                            margin: 0;
                            font-size: 1.2rem;
                        }
                        .crypto-details p {
                            color: #b9f9b9;
                            margin: 0 0 10px 0;
                            font-size: 0.9rem;
                        }
                        .crypto-features {
                            display: flex;
                            flex-wrap: wrap;
                            gap: 5px;
                        }
                        .feature-tag {
                            background: rgba(144, 238, 144, 0.2);
                            color: #90ee90;
                            padding: 4px 8px;
                            border-radius: 12px;
                            font-size: 0.75rem;
                            border: 1px solid rgba(144, 238, 144, 0.3);
                        }
                        .grok-outline {
                            display: grid;
                            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
                            gap: 15px;
                        }
                        .outline-item {
                            display: flex;
                            align-items: center;
                            gap: 12px;
                            padding: 15px;
                            background: rgba(144, 238, 144, 0.05);
                            border-radius: 10px;
                            border: 1px solid rgba(144, 238, 144, 0.2);
                        }
                        .outline-icon {
                            font-size: 1.2rem;
                        }
                        .outline-item h5 {
                            color: #90ee90;
                            margin: 0;
                            font-size: 1rem;
                        }
                        @keyframes grokFadeIn {
                            from { opacity: 0; }
                            to { opacity: 1; }
                        }
                        @keyframes grokSlideIn {
                            from { transform: translateY(-50px) scale(0.9); opacity: 0; }
                            to { transform: translateY(0) scale(1); opacity: 1; }
                        }
                        @keyframes grokFloat {
                            0%, 100% { transform: translateY(0px) rotate(0deg); }
                            50% { transform: translateY(-10px) rotate(1deg); }
                        }
                        @keyframes grokPulse {
                            0%, 100% { transform: scale(1); filter: drop-shadow(0 0 20px rgba(144, 238, 144, 0.6)); }
                            50% { transform: scale(1.05); filter: drop-shadow(0 0 30px rgba(144, 238, 144, 0.8)); }
                        }
                        @keyframes grokScan {
                            0% { transform: translateX(-100%); }
                            100% { transform: translateX(100%); }
                        }
                        .grok-modal-body::-webkit-scrollbar {
                            width: 8px;
                        }
                        .grok-modal-body::-webkit-scrollbar-track {
                            background: rgba(0, 0, 0, 0.3);
                            border-radius: 4px;
                        }
                        .grok-modal-body::-webkit-scrollbar-thumb {
                            background: linear-gradient(135deg, rgba(144, 238, 144, 0.6), rgba(144, 238, 144, 0.3));
                            border-radius: 4px;
                        }
                        .grok-modal-body::-webkit-scrollbar-thumb:hover {
                            background: linear-gradient(135deg, rgba(144, 238, 144, 0.8), rgba(144, 238, 144, 0.5));
                        }
                    </style>
                    ${guideContent}
                </div>

                <!-- Grok Footer -->
                <div class="grok-modal-footer" style="
                    background: linear-gradient(135deg, rgba(0, 0, 0, 0.5), rgba(20, 20, 40, 0.3));
                    border-top: 1px solid rgba(144, 238, 144, 0.2);
                    padding: 25px 30px;
                    border-radius: 0 0 18px 18px;
                    display: flex;
                    gap: 15px;
                    justify-content: flex-end;
                ">
                    <button onclick="closeModal()" style="
                        background: rgba(255, 255, 255, 0.1);
                        border: 1px solid rgba(255, 255, 255, 0.2);
                        color: #b9f9b9;
                        padding: 12px 24px;
                        border-radius: 25px;
                        cursor: pointer;
                        font-size: 0.9rem;
                        transition: all 0.3s ease;
                        backdrop-filter: blur(10px);
                    " onmouseover="this.style.background='rgba(255, 255, 255, 0.2)'; this.style.transform='translateY(-2px)'" 
                       onmouseout="this.style.background='rgba(255, 255, 255, 0.1)'; this.style.transform='translateY(0)'">
                        🔙 Voltar
                    </button>
                    <button onclick="requestFullGuide('${resource}')" style="
                        background: linear-gradient(135deg, rgba(144, 238, 144, 0.2), rgba(144, 238, 144, 0.3));
                        border: 1px solid rgba(144, 238, 144, 0.5);
                        color: #90ee90;
                        padding: 12px 24px;
                        border-radius: 25px;
                        cursor: pointer;
                        font-size: 0.9rem;
                        transition: all 0.3s ease;
                        backdrop-filter: blur(10px);
                        text-shadow: 0 0 10px rgba(144, 238, 144, 0.6);
                    " onmouseover="this.style.background='linear-gradient(135deg, rgba(144, 238, 144, 0.3), rgba(144, 238, 144, 0.4))'; this.style.transform='translateY(-2px) scale(1.05)'" 
                       onmouseout="this.style.background='linear-gradient(135deg, rgba(144, 238, 144, 0.2), rgba(144, 238, 144, 0.3))'; this.style.transform='translateY(0) scale(1)'">
                        📧 Solicitar Guia Completo
                    </button>
                </div>
            </div>
        `;

        document.body.appendChild(modal);
        console.log('✅ Modal Grok criado e adicionado ao DOM');

        // Fechar ao clicar fora
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                console.log('🔄 Fechando modal ao clicar fora');
                closeModal();
            }
        });

        // Animar entrada
        setTimeout(() => {
            modal.style.opacity = '1';
        }, 10);
    }

    // Modal de treinamento
    function showTrainingModal(resource, btn) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">🎓</div>
            <h3 style="color: #90ee90;">Treinamento: ${resource}</h3>
        </div>
        
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 20px; margin: 20px 0;">
            <h4>🏆 Programa de Treinamento:</h4>
            <ul style="text-align: left; margin: 10px 0;">
                <li><strong>Duração:</strong> 40-80 horas</li>
                <li><strong>Modalidade:</strong> Online/Presencial</li>
                <li><strong>Certificação:</strong> Incluída</li>
                <li><strong>Labs práticos:</strong> Ambiente real</li>
                <li><strong>Suporte:</strong> 6 meses</li>
            </ul>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            <i class="fas fa-times"></i> Fechar
        </button>
        <button class="modal-btn primary" onclick="openServiceModal('training-${resource}', 'Treinamento ${resource}', 'A partir de R$ 2.500')">
            <i class="fas fa-graduation-cap"></i> Solicitar Treinamento
        </button>`;

        openModal('🎓 Programa de Treinamento', content, footer);
    }

    // Modal de análise
    function showAnalysisModal(resource, btn) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">🔍</div>
            <h3 style="color: #90ee90;">Análise: ${resource}</h3>
        </div>
        
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 20px; margin: 20px 0;">
            <h4>📊 Relatório de Análise:</h4>
            <ul style="text-align: left; margin: 10px 0;">
                <li>Scan completo de vulnerabilidades</li>
                <li>Análise de indicadores de compromisso</li>
                <li>Relatório executivo e técnico</li>
                <li>Recomendações de mitigação</li>
                <li>Plano de ação prioritário</li>
            </ul>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            <i class="fas fa-times"></i> Fechar
        </button>
        <button class="modal-btn primary" onclick="openServiceModal('analysis-${resource}', 'Análise ${resource}', 'A partir de R$ 1.500')">
            <i class="fas fa-chart-line"></i> Solicitar Análise
        </button>`;

        openModal('🔍 Análise Profissional', content, footer);
    }

    // Modal para redes sociais
    function showSocialModal(btn) {
        const icon = btn.querySelector('i').className;
        const platform = icon.includes('twitter') ? 'Twitter' : 
                        icon.includes('github') ? 'GitHub' : 
                        icon.includes('linkedin') ? 'LinkedIn' : 'Social';
        
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">🌐</div>
            <h3 style="color: #90ee90;">Siga-nos no ${platform}</h3>
        </div>
        
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 20px; margin: 20px 0;">
            <p>🚀 <strong>Em breve:</strong> Nossa presença oficial no ${platform}!</p>
            <p>📢 Fique ligado para:</p>
            <ul style="text-align: left; margin: 10px 0;">
                <li>Updates de segurança em tempo real</li>
                <li>Novos tools e recursos</li>
                <li>Dicas e tutoriais exclusivos</li>
                <li>Comunidade de cybersecurity</li>
            </ul>
        </div>`;

        const footer = `
        <button class="modal-btn primary" onclick="closeModal()">
            <i class="fas fa-bell"></i> Entendi, vou aguardar!
        </button>`;

        openModal(`🌐 ${platform} - Em Breve`, content, footer);
    }

    // Modal genérico
    function showGenericInfoModal(resource, btn) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 3rem; margin-bottom: 15px;">ℹ️</div>
            <h3 style="color: #90ee90;">${resource}</h3>
        </div>
        
        <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 20px; margin: 20px 0;">
            <p>🚧 <strong>Recurso em desenvolvimento!</strong></p>
            <p>Esta funcionalidade está sendo preparada especialmente para você.</p>
            
            <h4>🎯 O que está por vir:</h4>
            <ul style="text-align: left; margin: 10px 0;">
                <li>Interface intuitiva e moderna</li>
                <li>Funcionalidades avançadas</li>
                <li>Integração com outras ferramentas</li>
                <li>Suporte completo</li>
            </ul>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            <i class="fas fa-times"></i> Fechar
        </button>
        <button class="modal-btn primary" onclick="requestNotification('${resource}')">
            <i class="fas fa-bell"></i> Me avisar quando disponível
        </button>`;

        openModal('ℹ️ Recurso em Desenvolvimento', content, footer);
    }

    // Funções auxiliares para ações dos modais
    window.notifyWhenReady = function(resource) {
        alert(`✅ Você será notificado quando "${resource}" estiver disponível!`);
        closeModal();
    };

    window.requestDemo = function(resource) {
        openServiceModal(`demo-${resource}`, `Demo ${resource}`, 'Gratuito');
    };

    window.subscribeGuide = function(resource) {
        alert(`📚 Você será notificado quando o guia "${resource}" for publicado!`);
        closeModal();
    };

    window.requestNotification = function(resource) {
        alert(`🔔 Você receberá uma notificação quando "${resource}" estiver pronto!`);
        closeModal();
    };

    // Solicitar guia completo
    window.requestFullGuide = function(resource) {
        const subject = `Solicitação - Guia Completo: ${resource}`;
        const body = `Olá Lucas,

Gostaria de receber o guia completo sobre "${resource}".

📚 GUIA SOLICITADO:
${resource}

💡 INTERESSE:
Preciso de informações detalhadas sobre este tópico para implementar em meu ambiente/projeto.

🎯 FORMATO PREFERIDO:
[ ] PDF detalhado
[ ] Vídeo tutorial
[ ] Consultoria ao vivo
[ ] Workshop prático

📧 CONTATO:
[Seu email aqui]

Aguardo retorno!

Atenciosamente,
[Seu nome]`;

        window.location.href = `mailto:lucasdeandrade0077@gmail.com?subject=${encodeURIComponent(subject)}&body=${encodeURIComponent(body)}`;
        closeModal();
    };

    // Modal de redirecionamento VPN
    window.openVPNRedirectModal = function(vpnName, promoCode) {
        const vpnUrls = {
            'NordVPN': 'https://nordvpn.com',
            'ExpressVPN': 'https://expressvpn.com',
            'Surfshark VPN': 'https://surfshark.com',
            'ProtonVPN': 'https://protonvpn.com'
        };
        
        const vpnUrl = vpnUrls[vpnName] || 'https://google.com/search?q=' + encodeURIComponent(vpnName);
        
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 4rem; margin-bottom: 15px; animation: pulse 2s infinite;">🚀</div>
            <h3 style="color: #90ee90; text-shadow: 0 0 15px rgba(144, 238, 144, 0.6);">Ativando ${vpnName}</h3>
        </div>
        
        <div style="background: linear-gradient(135deg, rgba(144, 238, 144, 0.05), rgba(144, 238, 144, 0.15)); border-radius: 15px; padding: 25px; margin: 20px 0; border: 2px solid rgba(144, 238, 144, 0.3);">
            <h4 style="color: #90ee90; margin-bottom: 15px;">🎁 Seu Código Promocional Exclusivo</h4>
            
            <div style="background: rgba(45, 80, 22, 0.8); padding: 20px; border-radius: 15px; text-align: center; margin: 20px 0; border: 2px solid rgba(144, 238, 144, 0.5);">
                <div style="color: #90ee90; font-size: 2rem; font-weight: bold; margin-bottom: 10px;">${promoCode}</div>
                <button onclick="copyToClipboard('${promoCode}')" style="
                    background: linear-gradient(135deg, rgba(144, 238, 144, 0.2), rgba(144, 238, 144, 0.3));
                    border: 1px solid rgba(144, 238, 144, 0.5);
                    color: #90ee90;
                    padding: 8px 16px;
                    border-radius: 10px;
                    cursor: pointer;
                    font-size: 0.9rem;
                ">📋 Copiar Código</button>
            </div>
            
            <div style="background: rgba(16, 185, 129, 0.1); border-radius: 10px; padding: 20px; margin: 15px 0;">
                <h5 style="color: #10b981; margin-bottom: 15px;">📋 Instruções:</h5>
                <ol style="color: #b9f9b9; line-height: 1.6;">
                    <li><strong>Clique em "Ir para ${vpnName}"</strong> abaixo</li>
                    <li><strong>Copie o código promocional</strong> ${promoCode}</li>
                    <li><strong>Cole no checkout</strong> para obter desconto</li>
                    <li><strong>Complete o cadastro</strong> e ative sua VPN</li>
                </ol>
            </div>
            
            <div style="background: rgba(255, 215, 0, 0.1); border-radius: 10px; padding: 15px; margin: 15px 0;">
                <p style="margin: 0; color: #fbbf24;"><strong>⚠️ Importante:</strong> Use o código no checkout para garantir o desconto!</p>
            </div>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            🔙 Cancelar
        </button>
        <button class="modal-btn primary" onclick="redirectToVPN('${vpnUrl}', '${vpnName}', '${promoCode}')">
            🚀 Ir para ${vpnName}
        </button>`;

        openModal(`🛡️ Ativando ${vpnName}`, content, footer);
    };

    // Redirecionar para VPN
    window.redirectToVPN = function(url, vpnName, promoCode) {
        // Copiar código automaticamente
        copyToClipboard(promoCode);
        
        // Mostrar confirmação
        const content = `
        <div style="text-align: center;">
            <div style="font-size: 4rem; margin-bottom: 20px; color: #10b981;">✅</div>
            <h3 style="color: #10b981;">Código Copiado!</h3>
            <p style="color: #b9f9b9; margin: 20px 0;">
                O código <strong>${promoCode}</strong> foi copiado para sua área de transferência.
            </p>
            <p style="color: #90ee90;">
                Você será redirecionado para ${vpnName} em 3 segundos...
            </p>
        </div>`;

        openModal('🎉 Redirecionando...', content, '');
        
        // Redirecionar após 3 segundos
        setTimeout(() => {
            window.open(url, '_blank');
            closeModal();
        }, 3000);
    };

    // Simulador de download inovador
    window.startFakeDownload = function(resource) {
        const statusEl = document.getElementById('download-status');
        const steps = [
            'Verificando integridade...',
            'Conectando ao servidor...',
            'Iniciando download...',
            'Baixando arquivos...',
            'Verificando checksums...',
            'Download concluído!'
        ];
        
        let step = 0;
        const interval = setInterval(() => {
            if (statusEl && step < steps.length) {
                statusEl.textContent = steps[step];
                step++;
            } else {
                clearInterval(interval);
                setTimeout(() => {
                    showDownloadComplete(resource);
                }, 500);
            }
        }, 500);
    };

    // Modal de download completo
    function showDownloadComplete(resource) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 4rem; margin-bottom: 15px; color: #10b981;">✅</div>
            <h3 style="color: #10b981; text-shadow: 0 0 15px rgba(16, 185, 129, 0.6);">Download Simulado Concluído!</h3>
        </div>
        
        <div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.1), rgba(16, 185, 129, 0.2)); border-radius: 15px; padding: 25px; margin: 20px 0; border: 2px solid rgba(16, 185, 129, 0.4);">
            <h4 style="color: #10b981; margin-bottom: 15px;">🎉 ${resource} baixado com sucesso!</h4>
            <p style="color: #b9f9b9; margin-bottom: 15px;">Em um cenário real, você teria acesso a:</p>
            <ul style="text-align: left; margin: 10px 0; color: #b9f9b9;">
                <li>• 📁 Pasta com código fonte</li>
                <li>• 📖 Manual de instalação</li>
                <li>• 🔧 Scripts de configuração</li>
                <li>• 📋 Exemplos práticos</li>
                <li>• 🎯 Guia de uso avançado</li>
            </ul>
            
            <div style="background: rgba(144, 238, 144, 0.1); border-radius: 10px; padding: 15px; margin: 15px 0;">
                <p style="margin: 0;"><strong>💡 Dica:</strong> Entre em contato para acessar os recursos reais!</p>
            </div>
        </div>`;

        const footer = `
        <button class="modal-btn primary" onclick="closeModal()">
            🚀 Entendi!
        </button>`;

        openModal('✅ Download Concluído', content, footer);
    }

    // Modal de VPN inovador para privacidade
    window.openAdvancedVPNModal = function(vpnName) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 4rem; margin-bottom: 15px; animation: pulse 2s infinite;">🛡️</div>
            <h3 style="color: #90ee90; text-shadow: 0 0 15px rgba(144, 238, 144, 0.6);">Configuração Avançada: ${vpnName}</h3>
        </div>
        
        <!-- Teste de velocidade simulado -->
        <div style="background: linear-gradient(135deg, rgba(144, 238, 144, 0.05), rgba(144, 238, 144, 0.15)); border-radius: 15px; padding: 25px; margin: 20px 0; border: 2px solid rgba(144, 238, 144, 0.3);">
            <h4 style="color: #90ee90; margin-bottom: 15px;">⚡ Teste de Velocidade</h4>
            <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 15px; margin: 15px 0;">
                <div style="text-align: center; padding: 15px; background: rgba(144, 238, 144, 0.1); border-radius: 10px;">
                    <div style="color: #90ee90; font-size: 1.5rem; font-weight: bold;">95 Mbps</div>
                    <div style="color: #b9f9b9; font-size: 0.8rem;">Download</div>
                </div>
                <div style="text-align: center; padding: 15px; background: rgba(144, 238, 144, 0.1); border-radius: 10px;">
                    <div style="color: #90ee90; font-size: 1.5rem; font-weight: bold;">87 Mbps</div>
                    <div style="color: #b9f9b9; font-size: 0.8rem;">Upload</div>
                </div>
                <div style="text-align: center; padding: 15px; background: rgba(144, 238, 144, 0.1); border-radius: 10px;">
                    <div style="color: #90ee90; font-size: 1.5rem; font-weight: bold;">12 ms</div>
                    <div style="color: #b9f9b9; font-size: 0.8rem;">Ping</div>
                </div>
            </div>
            
            <!-- Servidores disponíveis -->
            <h5 style="color: #90ee90; margin: 20px 0 10px 0;">🌍 Servidores Disponíveis:</h5>
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 10px;">
                <div style="padding: 10px; background: rgba(144, 238, 144, 0.1); border-radius: 8px; display: flex; justify-content: space-between;">
                    <span style="color: #b9f9b9;">🇧🇷 São Paulo</span>
                    <span style="color: #10b981;">●</span>
                </div>
                <div style="padding: 10px; background: rgba(144, 238, 144, 0.1); border-radius: 8px; display: flex; justify-content: space-between;">
                    <span style="color: #b9f9b9;">🇺🇸 Miami</span>
                    <span style="color: #10b981;">●</span>
                </div>
                <div style="padding: 10px; background: rgba(144, 238, 144, 0.1); border-radius: 8px; display: flex; justify-content: space-between;">
                    <span style="color: #b9f9b9;">🇳🇱 Amsterdam</span>
                    <span style="color: #f7931e;">●</span>
                </div>
                <div style="padding: 10px; background: rgba(144, 238, 144, 0.1); border-radius: 8px; display: flex; justify-content: space-between;">
                    <span style="color: #b9f9b9;">🇯🇵 Tokyo</span>
                    <span style="color: #10b981;">●</span>
                </div>
            </div>
        </div>`;

        const footer = `
        <button class="modal-btn secondary" onclick="closeModal()">
            🔙 Voltar
        </button>
        <button class="modal-btn primary" onclick="simulateVPNConnection('${vpnName}')">
            🔗 Conectar Agora
        </button>`;

        openModal(`🛡️ ${vpnName} - Painel Avançado`, content, footer);
    };

    // Simulador de conexão VPN
    window.simulateVPNConnection = function(vpnName) {
        const content = `
        <div style="text-align: center;">
            <div style="font-size: 4rem; margin-bottom: 20px; animation: spin 2s linear infinite;">🔄</div>
            <h4 style="color: #90ee90;">Conectando ao ${vpnName}...</h4>
            <div style="width: 100%; height: 6px; background: rgba(144, 238, 144, 0.3); border-radius: 3px; overflow: hidden; margin: 20px 0;">
                <div style="width: 0%; height: 100%; background: linear-gradient(90deg, #90ee90, #10b981); border-radius: 3px; animation: connectionProgress 4s ease-out forwards;"></div>
            </div>
            <div id="connection-status" style="color: #b9f9b9; margin-top: 15px;">Estabelecendo túnel seguro...</div>
        </div>`;

        openModal('🔗 Conectando VPN...', content, '');

        // Simular etapas de conexão
        const statuses = [
            'Verificando credenciais...',
            'Selecionando servidor ótimo...',
            'Estabelecendo túnel criptografado...',
            'Configurando DNS seguro...',
            'Ativando kill switch...',
            'Conexão estabelecida com sucesso!'
        ];

        let step = 0;
        const interval = setInterval(() => {
            const statusEl = document.getElementById('connection-status');
            if (statusEl && step < statuses.length) {
                statusEl.textContent = statuses[step];
                step++;
            } else {
                clearInterval(interval);
                setTimeout(() => {
                    showVPNConnected(vpnName);
                }, 800);
            }
        }, 600);
    };

    // VPN conectada com sucesso
    function showVPNConnected(vpnName) {
        const content = `
        <div style="text-align: center; margin-bottom: 25px;">
            <div style="font-size: 4rem; margin-bottom: 15px; color: #10b981;">🛡️</div>
            <h3 style="color: #10b981; text-shadow: 0 0 15px rgba(16, 185, 129, 0.6);">Conectado ao ${vpnName}!</h3>
        </div>
        
        <div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.1), rgba(16, 185, 129, 0.2)); border-radius: 15px; padding: 25px; margin: 20px 0; border: 2px solid rgba(16, 185, 129, 0.4);">
            <h4 style="color: #10b981; margin-bottom: 15px;">🔐 Status da Conexão</h4>
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div style="background: rgba(0,0,0,0.3); padding: 15px; border-radius: 10px;">
                    <div style="color: #10b981; font-weight: bold;">IP Público:</div>
                    <div style="color: #b9f9b9; font-family: monospace;">185.243.218.45</div>
                </div>
                <div style="background: rgba(0,0,0,0.3); padding: 15px; border-radius: 10px;">
                    <div style="color: #10b981; font-weight: bold;">Localização:</div>
                    <div style="color: #b9f9b9;">🇳🇱 Amsterdam, NL</div>
                </div>
                <div style="background: rgba(0,0,0,0.3); padding: 15px; border-radius: 10px;">
                    <div style="color: #10b981; font-weight: bold;">Protocolo:</div>
                    <div style="color: #b9f9b9;">WireGuard</div>
                </div>
                <div style="background: rgba(0,0,0,0.3); padding: 15px; border-radius: 10px;">
                    <div style="color: #10b981; font-weight: bold;">Criptografia:</div>
                    <div style="color: #b9f9b9;">AES-256</div>
                </div>
            </div>
            
            <div style="margin-top: 20px; padding: 15px; background: rgba(16, 185, 129, 0.1); border-radius: 10px;">
                <p style="margin: 0; color: #10b981;"><strong>✅ Você está protegido!</strong></p>
                <p style="margin: 5px 0 0 0; color: #b9f9b9; font-size: 0.9rem;">Sua conexão está criptografada e seu IP está oculto.</p>
            </div>
        </div>`;

        const footer = `
        <button class="modal-btn primary" onclick="closeModal()">
            🌟 Perfeito!
        </button>`;

        openModal('✅ VPN Ativa', content, footer);
    }

})();
