/**
 * Sistema de Notícias Eficaz - Carrega primeiro, depois fixa
 * Sistema otimizado que puxa de múltiplas fontes simultaneamente
 */
class NewsSystem {
    constructor() {
        this.currentPage = 1;
        this.hasMoreNews = true;
        this.isLoading = false;
        this.currentLanguage = 'all';
        this.loadedIds = new Set();
        this.cache = new Map();
        this.usedImages = new Set();
        this.imageKeywords = this.initializeImageKeywords();
        
        console.log('🚀 Iniciando sistema de notícias eficaz...');
        this.init();
    }
    
    async init() {
        console.log('🚀 Iniciando sistema NewsSystem...');
        try {
            
            await this.loadInitialNews();
            
            
            this.setupFilters();
            this.setupInfiniteScroll();
            
            console.log('🎠 Chamando setupCarousel...');
            this.setupCarousel();
            
            console.log('✅ Sistema inicializado com sucesso!');
            
        } catch (error) {
            console.error('❌ Erro na inicialização:', error);
        }
    }
    
    async loadInitialNews() {
        console.log('📰 Carregando notícias iniciais...');
        
        try {
            const response = await fetch(`/backend/api/index.php/articles?page=1&limit=40&_t=${Date.now()}`);
            const data = await response.json();
            
            if (data.success && data.data && data.data.length > 0) {
                await this.renderNews(data.data, true);
                this.currentPage = 2;
                this.hasMoreNews = data.pagination && data.pagination.has_next;
                
                console.log(`✅ ${data.data.length} notícias carregadas e FIXAS!`);
            } else {
                console.log('⚠️  Nenhuma notícia encontrada, executando scraper...');
                await this.runScraper();
            }
            
        } catch (error) {
            console.error('❌ Erro ao carregar notícias:', error);
            await this.runScraper();
        }
    }
    
    async runScraper() {
        console.log('🔄 Executando scraper multi-fonte...');
        
        try {
            const response = await fetch('/backend/scraper/run-scraper.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                }
            });
            
            const result = await response.json();
            console.log('📊 Scraper executado:', result);
            
            if (result.success) {
                console.log(`✅ Scraper coletou ${result.total_extracted} artigos`);
                
                setTimeout(() => this.loadInitialNews(), 3000);
            } else {
                console.log('⚠️  Scraper falhou:', result.message);
                this.loadFallbackNews();
            }
            
        } catch (error) {
            console.error('❌ Erro no scraper:', error);
            this.loadFallbackNews();
        }
    }
    
    loadFallbackNews() {
        console.log('🆘 Carregando notícias de fallback...');
        
        const fallbackNews = [
            {
                id: 1,
                title: 'Sistema de Notícias Carregado com Sucesso',
                description: 'O novo sistema de agregação de notícias está funcionando corretamente.',
                url: '#',
                source_name: 'Sistema',
                language: 'pt-BR',
                published_at: new Date().toISOString(),
                views_count: 0,
                image_url: null
            }
        ];
        
        this.renderNews(fallbackNews, true);
    }
    
    async renderNews(articles, isInitial = false) {
        const newsGrid = document.getElementById('news-grid');
        if (!newsGrid) return;
        
        if (isInitial) {
            newsGrid.innerHTML = '';
            this.loadedIds.clear();
        }
        
        const newCards = [];
        
        for (const article of articles) {
            if (this.loadedIds.has(article.id)) continue;
            
            this.loadedIds.add(article.id);
            const card = this.createNewsCard(article);
            
            
            card.style.opacity = '0';
            card.style.transform = 'translateY(30px)';
            
            newsGrid.appendChild(card);
            newCards.push(card);
        }
        
        
        for (let i = 0; i < newCards.length; i++) {
            const card = newCards[i];
            
            setTimeout(() => {
                card.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
            }, i * 150);
        }
        
        
        if (!isInitial) {
            setTimeout(() => {
                this.resolveImagesForNewArticles(articles);
            }, 200);
        }
        
        
        if (this.imageObserver) {
            newCards.forEach(card => {
                const img = card.querySelector('img');
                if (img) {
                    this.imageObserver.observe(img);
                }
            });
        }
        
        console.log(`✨ ${newCards.length} cards renderizados e FIXOS`);
    }
    
    createNewsCard(article) {
        const card = document.createElement('article');
        card.className = 'news-card cannabis-card';
        card.setAttribute('data-news-id', article.id);
        
        const langClass = article.language.includes('pt') ? 'lang-pt' : 'lang-en';
        const langText = article.language.includes('pt') ? 'PT' : 'EN';
        const langColor = article.language.includes('pt') ? '#228B22' : '#8B0000';
        
        card.innerHTML = `
            <div class="language-indicator ${langClass}" style="background: ${langColor}; position: absolute; top: 10px; left: 10px; z-index: 10; padding: 4px 8px; border-radius: 12px; font-weight: bold; font-size: 10px; color: white;">
                ${langText}
            </div>
            <div class="news-image">
                <img src="" 
                     alt="${article.title}" 
                     loading="lazy" 
                     style="opacity: 0; transition: opacity 0.5s ease;"
                     data-article-id="${article.id}">
                <div class="image-placeholder show">
                    <div style="text-align: center;">
                        <i class="fas fa-image" style="font-size: 2rem; margin-bottom: 10px; opacity: 0.7;"></i>
                        <div>Imagem não disponível</div>
                    </div>
                </div>
            </div>
            <div class="news-content">
                <div class="news-meta">
                    <span class="source-name">${article.source_name || 'Fonte'}</span>
                    <span class="publish-date">${this.formatDate(article.published_at)}</span>
                </div>
                <h3 class="news-title">${article.title}</h3>
                <p class="news-description">${this.escapeHtml(this.truncateDescription(this.stripHtml(article.description), 25))}</p>
                <div class="news-footer">
                    <a href="${article.url}" 
                       class="read-more cannabis-button" 
                       target="_blank" 
                       rel="noopener noreferrer"
                       onclick="newsSystem.trackClick(${article.id})"
                       style="background: rgba(34, 139, 34, 0.8); color: white; border-color: rgba(34, 139, 34, 0.8);">
                        <span>Ler notícia</span>
                        <i class="fas fa-external-link-alt"></i>
                    </a>
                    <div class="news-stats">
                        <span class="news-date">
                            <i class="fas fa-calendar-alt"></i>
                            <span class="date">${this.formatDate(article.published_at)}</span>
                        </span>
                    </div>
                </div>
            </div>
        `;
        
        
        const img = card.querySelector('img');
        const placeholder = card.querySelector('.image-placeholder');
        
        if (img) {
            const self = this;
            
            img.onload = function() {
                
                this.style.opacity = '1';
                if (placeholder) placeholder.classList.remove('show');
            };
            
            img.onerror = async function() {
                const articleId = this.getAttribute('data-article-id');
                console.log(`❌ Erro ao carregar imagem para artigo: ${articleId}`);
                if (placeholder) placeholder.classList.add('show');
                
                try {
                    const params = new URLSearchParams({
                        article_id: articleId,
                        title: article.title || ''
                    });
                    const proxyResp = await fetch(`/backend/api/image_proxy.php?${params}`, { cache: 'no-store' });
                    if (proxyResp.ok) {
                        const proxyData = await proxyResp.json();
                        if (proxyData.success && proxyData.data && proxyData.data.local_url) {
                            this.onerror = null; 
                            this.src = proxyData.data.local_url + `?t=${Date.now()}`;
                            this.style.display = '';
                            return;
                        }
                    }
                } catch (e) {
                    
                }
                
                this.style.display = 'none';
            };
        }
        
        
        this.resolveAndSetImage(article, img, placeholder);
        
        return card;
    }
    async resolveAndSetImage(article, imgElement, placeholder) {
        if (!imgElement) return;
        
        if (article.image_url && article.image_url.trim() !== '' && article.image_url !== 'null') {
            
            try {
                const params = new URLSearchParams({ url: article.image_url });
                const proxyResp = await fetch(`/backend/api/image_proxy.php?${params}`, { cache: 'no-store' });
                if (proxyResp.ok) {
                    const proxyData = await proxyResp.json();
                    if (proxyData.success && proxyData.data && proxyData.data.local_url) {
                        imgElement.src = proxyData.data.local_url + `?t=${Date.now()}`;
                        imgElement.onload = function() {
                            this.style.opacity = '1';
                            if (placeholder) placeholder.classList.remove('show');
                        };
                        imgElement.onerror = function() {
                            if (placeholder) placeholder.classList.add('show');
                            this.style.display = 'none';
                        };
                        return;
                    }
                }
            } catch (e) {}
            
            imgElement.src = article.image_url;
            imgElement.onload = function() {
                this.style.opacity = '1';
                if (placeholder) placeholder.classList.remove('show');
            };
            imgElement.onerror = function() {
                
                if (placeholder) placeholder.classList.add('show');
                this.style.display = 'none';
            };
            return;
        }
        
        if (article.url && article.image_url) {
            try {
                const params = new URLSearchParams({ url: article.image_url });
                const proxyResp = await fetch(`/backend/api/image_proxy.php?${params}`, { cache: 'no-store' });
                if (proxyResp.ok) {
                    const proxyData = await proxyResp.json();
                    if (proxyData.success && proxyData.data && proxyData.data.local_url) {
                        imgElement.onerror = null;
                        imgElement.src = proxyData.data.local_url + `?t=${Date.now()}`;
                        imgElement.style.display = '';
                        if (placeholder) placeholder.classList.remove('show');
                        return;
                    }
                }
            } catch (e) {}
        }
        
        if (placeholder) placeholder.classList.add('show');
        imgElement.style.display = 'none';
    }
    
    
    resolveImagesForNewArticles(articles) {
        console.log(`🖼️ Resolvendo imagens para ${articles.length} novos artigos...`);
        
        articles.forEach(article => {
            const card = document.querySelector(`[data-news-id="${article.id}"]`);
            if (card) {
                const img = card.querySelector('img');
                const placeholder = card.querySelector('.image-placeholder');
                
                if (img && !img.src) {
                    console.log(`🔄 Resolvendo imagem para artigo ${article.id}: ${article.title}`);
                    this.resolveAndSetImage(article, img, placeholder);
                }
            }
        });
        
        
        setTimeout(() => {
            this.retryFailedImages(articles);
        }, 2000);
    }
    
    
    retryFailedImages(articles) {
        console.log('🔄 Verificando imagens que falharam...');
        
        articles.forEach(article => {
            const card = document.querySelector(`[data-news-id="${article.id}"]`);
            if (card) {
                const img = card.querySelector('img');
                const placeholder = card.querySelector('.image-placeholder');
                
                
                if (img && (!img.src || img.src === '')) {
                    console.log(`🔄 Mostrando placeholder para artigo ${article.id}`);
                    if (placeholder) placeholder.classList.add('show');
                    img.style.display = 'none';
                }
            }
        });
    }
    
    
    handleImageError(imgElement, articleId) {
        console.log(`🔄 Erro ao carregar imagem para artigo ${articleId}, mostrando placeholder`);
        
        
                imgElement.style.display = 'none';
                const placeholder = imgElement.parentElement.querySelector('.image-placeholder');
                if (placeholder) {
                    placeholder.classList.add('show');
                }
    }
    
    formatDate(dateString) {
        const date = new Date(dateString);
        const now = new Date();
        const diffTime = Math.abs(now - date);
        const diffHours = Math.ceil(diffTime / (1000 * 60 * 60));
        
        if (diffHours < 1) return 'Agora';
        if (diffHours < 24) return `${diffHours}h atrás`;
        if (diffHours < 48) return 'Ontem';
        
        return date.toLocaleDateString('pt-BR');
    }
    
    
    trackClick(articleId) {
        fetch('/backend/api/index.php/view', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ article_id: articleId })
        }).catch(e => console.log('Erro ao rastrear clique:', e));
    }
    
    setupFilters() {
        const filterBtns = document.querySelectorAll('.filter-btn');
        
        filterBtns.forEach(btn => {
            btn.addEventListener('click', () => {
                filterBtns.forEach(b => b.classList.remove('active'));
                btn.classList.add('active');
                
                const filter = btn.getAttribute('data-filter');
                this.currentLanguage = filter;
                this.currentPage = 1;
                this.hasMoreNews = true;
                
                this.loadInitialNews();
            });
        });
    }
    
    setupInfiniteScroll() {
        let scrollTimeout;
        let lastScrollTime = 0;
        
        window.addEventListener('scroll', () => {
            clearTimeout(scrollTimeout);
            scrollTimeout = setTimeout(() => {
                const now = Date.now();
                const scrollTop = window.pageYOffset;
                const windowHeight = window.innerHeight;
                const documentHeight = document.documentElement.scrollHeight;
                
                if (scrollTop + windowHeight >= documentHeight - 1200 && 
                    this.hasMoreNews && 
                    !this.isLoading &&
                    now - lastScrollTime > 1000) {
                    
                    lastScrollTime = now;
                    this.loadMoreNews();
                }
                
                
                this.checkVisibleImages();
            }, 400);
        });
        
        
        this.setupImageObserver();
    }
    
    setupImageObserver() {
        
        const imageObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    const articleId = img.getAttribute('data-article-id');
                    
                    
                    if (img && (!img.src || img.src === '')) {
                        console.log(`👁️ Imagem visível detectada para artigo ${articleId}, resolvendo...`);
                        this.resolveImageForVisibleArticle(articleId, img);
                    }
                }
            });
        }, {
            rootMargin: '50px' 
        });
        
        
        document.querySelectorAll('.news-card img').forEach(img => {
            imageObserver.observe(img);
        });
        
        
        this.imageObserver = imageObserver;
    }
    
    resolveImageForVisibleArticle(articleId, imgElement) {
        
        const card = document.querySelector(`[data-news-id="${articleId}"]`);
        if (!card) return;
        
        const placeholder = card.querySelector('.image-placeholder');
        
        
        if (placeholder) placeholder.classList.add('show');
        imgElement.style.display = 'none';
    }
    
    checkVisibleImages() {
        
        const visibleImages = document.querySelectorAll('.news-card img');
        visibleImages.forEach(img => {
            const rect = img.getBoundingClientRect();
            const isVisible = rect.top < window.innerHeight && rect.bottom > 0;
            
            if (isVisible && (!img.src || img.src === '')) {
                const articleId = img.getAttribute('data-article-id');
                console.log(`👁️ Imagem visível sem src detectada para artigo ${articleId}`);
                
                
                const card = document.querySelector(`[data-news-id="${articleId}"]`);
                if (card) {
                    const placeholder = card.querySelector('.image-placeholder');
                    if (placeholder) placeholder.classList.add('show');
                    img.style.display = 'none';
                }
            }
        });
    }
    
    async loadMoreNews() {
        if (this.isLoading || !this.hasMoreNews) return;
        
        this.isLoading = true;
        console.log(`🔄 Carregando página ${this.currentPage}...`);
        
        try {
            const params = new URLSearchParams({
                page: this.currentPage,
                limit: 40
            });
            
            if (this.currentLanguage !== 'all') {
                params.append('language', this.currentLanguage === 'pt' ? 'pt-BR' : 'en-US');
            }
            
            const response = await fetch(`/backend/api/index.php/articles?${params}&_t=${Date.now()}`);
            const data = await response.json();
            
            if (data.success && data.data && data.data.length > 0) {
                
                await this.renderNews(data.data);
                
                
                setTimeout(() => {
                    this.resolveImagesForNewArticles(data.data);
                }, 100);
                
                this.currentPage++;
                this.hasMoreNews = data.pagination && data.pagination.has_next;
                
                console.log(`✅ Página ${this.currentPage - 1} carregada com ${data.data.length} notícias`);
            } else {
                this.hasMoreNews = false;
                console.log('📄 Fim das notícias');
            }
            
        } catch (error) {
            console.error('❌ Erro ao carregar mais notícias:', error);
        } finally {
            this.isLoading = false;
        }
    }
    
    async setupCarousel() {
        console.log('🎠 Iniciando setup do carousel...');
        
        try {
            const response = await fetch('/backend/api/index.php/featured?limit=5');
            const data = await response.json();
            
            console.log('📊 Dados da API:', data);
            
            if (data.success && data.data && data.data.length > 0) {
                console.log(`✅ Encontradas ${data.data.length} notícias em destaque`);
                this.createCarousel(data.data);
            } else {
                console.log('⚠️  Sem notícias em destaque');
                this.hideCarousel();
            }
        } catch (error) {
            console.error('❌ Erro no carousel:', error);
            this.hideCarousel();
        }
    }
    
    createTestCarousel() {
        const testData = [
            {
                id: 1,
                title: "🌿 Teste: Nova tecnologia de segurança canábica",
                description: "Esta é uma notícia de teste para verificar se o carousel está funcionando corretamente com tema canábico.",
                url: "#",
                source_name: "CyberNews Teste",
                published_at: "2025-01-08 10:00:00",
                language: "pt-BR",
                image_url: null
            },
            {
                id: 2,
                title: "🇺🇸 Test: New cybersecurity framework",
                description: "This is a test news to verify if the carousel is working properly with english content.",
                url: "#",
                source_name: "Test Source",
                published_at: "2025-01-08 09:30:00",
                language: "en-US",
                image_url: null
            },
            {
                id: 3,
                title: "🌱 Teste: Proteção de dados inovadora",
                description: "Terceira notícia de teste para verificar a navegação entre slides do carousel canábico.",
                url: "#",
                source_name: "Verde Seguro",
                published_at: "2025-01-08 09:00:00",
                language: "pt-BR",
                image_url: null
            }
        ];
        
        console.log('🧪 Criando carousel de teste com', testData.length, 'slides');
        this.createCarousel(testData);
    }
    
    hideCarousel() {
        console.log('🚫 Ocultando carousel - sem notícias em destaque');
        const carousel = document.querySelector('.featured-carousel');
        if (carousel) {
            
            const track = document.getElementById('carousel-track');
            if (track) {
                track.innerHTML = `
                    <div style="display: flex; align-items: center; justify-content: center; height: 450px; background: linear-gradient(135deg, #2d5016, #4a7c23); color: #90ee90; border-radius: 15px; box-shadow: 0 0 20px rgba(0, 255, 100, 0.3);">
                        <div style="text-align: center;">
                            <i class="fas fa-seedling" style="font-size: 4rem; margin-bottom: 20px; color: #90ee90; text-shadow: 0 0 10px rgba(144, 238, 144, 0.5);"></i>
                            <h3 style="color: #90ee90; text-shadow: 0 0 5px rgba(144, 238, 144, 0.3); margin-bottom: 15px;">🌿 Nenhuma notícia em destaque encontrada</h3>
                            <p style="color: rgba(144, 238, 144, 0.8);">🌱 Execute o scraper para cultivar notícias frescas</p>
                        </div>
                    </div>
                `;
            }
        }
    }
    
    createCarousel(articles) {
        console.log('🔧 Criando carousel com', articles.length, 'artigos');
        
        const track = document.getElementById('carousel-track');
        const dotsContainer = document.getElementById('carousel-dots');
        
        console.log('🎯 Elementos encontrados:', {
            track: !!track,
            dotsContainer: !!dotsContainer
        });
        
        if (!track) {
            console.error('❌ Elemento carousel-track não encontrado!');
            return;
        }
        
        if (!dotsContainer) {
            console.error('❌ Elemento carousel-dots não encontrado!');
            return;
        }
        
        if (!articles.length) {
            console.error('❌ Nenhum artigo fornecido!');
            return;
        }
        
        
        track.innerHTML = '';
        dotsContainer.innerHTML = '';
        
        
        articles.forEach((article, index) => {
            console.log(`📰 Criando slide ${index + 1}:`, article.title);
            this.createSlide(article, index, track);
            this.createDot(index, dotsContainer);
        });
        
        
        const firstSlide = track.querySelector('.carousel-item');
        if (firstSlide) {
            firstSlide.classList.add('active');
        }
        
        const firstDot = dotsContainer.querySelector('.carousel-dot');
        if (firstDot) {
            firstDot.classList.add('active');
        }
        
        
        this.initNewCarouselControls(articles.length);
        
        console.log('✅ Carousel criado com sucesso!');
    }
    
    createSlide(article, index, container) {
        console.log(`🔨 Criando slide ${index}:`, article.title);
        
        const slide = document.createElement('div');
        slide.className = `carousel-item ${index === 0 ? 'active' : ''}`;
        
        console.log(`📐 Slide ${index} classes:`, slide.className);
        
        
        if (article.image_url && article.image_url.trim() !== '' && article.image_url !== 'null') {
            slide.style.backgroundImage = `url("${article.image_url}")`;
            console.log(`🖼️  Slide ${index} com imagem original:`, article.image_url);
        } else {
            
            slide.style.backgroundColor = '#2d5016';
            console.log(`🎨 Slide ${index} com cor sólida`);
        }
        
        const langClass = article.language.includes('pt') ? 'pt' : 'en';
        const langText = article.language.includes('pt') ? '🇧🇷 PT' : '🇺🇸 EN';
        
        slide.innerHTML = `
            <div class="language-tag ${langClass}">${langText}</div>
            <div class="carousel-content">
                <div class="carousel-meta">
                    <span class="carousel-source" style="color: #90ee90;">🌿 ${article.source_name}</span>
                    <span class="carousel-date" style="color: rgba(144, 238, 144, 0.8);">📅 ${this.formatDate(article.published_at)}</span>
                </div>
                <h2 class="carousel-title">${this.escapeHtml(article.title)}</h2>
                <p class="carousel-description">${this.escapeHtml(this.truncateDescription(article.description, 35))}</p>
                <a href="${article.url}" 
                   class="carousel-link" 
                   target="_blank"
                   rel="noopener noreferrer"
                   onclick="newsSystem.trackClick(${article.id})">
                    <span>🌿 Ler Notícia</span>
                    <i class="fas fa-external-link-alt"></i>
                </a>
            </div>
        `;
        
        container.appendChild(slide);
        console.log(`✅ Slide ${index} adicionado ao container`);
    }
    
    createDot(index, container) {
        const dot = document.createElement('div');
        dot.className = `carousel-dot ${index === 0 ? 'active' : ''}`;
        dot.addEventListener('click', () => this.goToSlide(index));
        container.appendChild(dot);
    }
    
    initNewCarouselControls(totalSlides) {
        const prevBtn = document.getElementById('carousel-prev');
        const nextBtn = document.getElementById('carousel-next');
        
        this.currentSlideIndex = 0;
        this.totalSlides = totalSlides;
        
        if (prevBtn) {
            prevBtn.addEventListener('click', () => this.prevSlide());
        }
        
        if (nextBtn) {
            nextBtn.addEventListener('click', () => this.nextSlide());
        }
        
        
        this.startAutoPlay();
    }
    
    goToSlide(index) {
        const slides = document.querySelectorAll('.carousel-item');
        const dots = document.querySelectorAll('.carousel-dot');
        
        
        slides.forEach(slide => slide.classList.remove('active'));
        dots.forEach(dot => dot.classList.remove('active'));
        
        
        if (slides[index]) slides[index].classList.add('active');
        if (dots[index]) dots[index].classList.add('active');
        
        this.currentSlideIndex = index;
    }
    
    nextSlide() {
        const nextIndex = (this.currentSlideIndex + 1) % this.totalSlides;
        this.goToSlide(nextIndex);
    }
    
    prevSlide() {
        const prevIndex = (this.currentSlideIndex - 1 + this.totalSlides) % this.totalSlides;
        this.goToSlide(prevIndex);
    }
    
    startAutoPlay() {
        
        if (this.autoPlayInterval) {
            clearInterval(this.autoPlayInterval);
        }
        
        
        this.autoPlayInterval = setInterval(() => {
            this.nextSlide();
        }, 8000); 
        
        
        const carousel = document.querySelector('.featured-carousel');
        if (carousel) {
            carousel.addEventListener('mouseenter', () => {
                if (this.autoPlayInterval) {
                    clearInterval(this.autoPlayInterval);
                }
            });
            
            carousel.addEventListener('mouseleave', () => {
                this.startAutoPlay();
            });
        }
    }
    
    escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    stripHtml(text) {
        if (!text || typeof text !== 'string') return '';
        
        let clean = text.replace(/<img[^>]*>/gi, '');
        
        clean = clean.replace(/<[^>]+>/g, '');
        
        clean = clean.replace(/\s+/g, ' ').trim();
        return clean;
    }
    
    initializeImageKeywords() {
        return {
            
            'hack|attack|ataque|breach|invasion': [
                'cybersecurity attack', 'hacker computer', 'data breach security', 'cyber attack network', 'computer hacking'
            ],
            
            
            'ransomware|virus|malware|trojan': [
                'ransomware computer virus', 'malware security threat', 'computer virus attack', 'cybersecurity malware', 'trojan horse virus'
            ],
            
            
            'leak|vazamento|data|dados|breach|exposure': [
                'data leak security', 'information breach', 'data privacy violation', 'confidential data exposed', 'database leak'
            ],
            
            
            'vulnerability|vulnerabilidade|exploit|zero-day|cve': [
                'security vulnerability', 'software exploit', 'zero day attack', 'system vulnerability', 'security flaw'
            ],
            
            
            'phishing|scam|fraud|social engineering|engenharia social': [
                'phishing email attack', 'social engineering scam', 'online fraud security', 'email phishing threat', 'cyber fraud'
            ],
            
            
            'crypto|bitcoin|blockchain|cryptocurrency|nft': [
                'cryptocurrency security', 'bitcoin blockchain', 'crypto wallet hack', 'blockchain technology', 'digital currency'
            ],
            
            
            'ai|artificial intelligence|machine learning|deep learning|neural': [
                'artificial intelligence security', 'AI machine learning', 'neural network technology', 'deep learning AI', 'AI cybersecurity'
            ],
            
            
            'google|microsoft|apple|meta|facebook|twitter|amazon': [
                'tech company security', 'big tech cybersecurity', 'technology corporation', 'silicon valley tech', 'tech industry'
            ],
            
            
            'network|rede|server|servidor|cloud|nuvem': [
                'network security infrastructure', 'server cybersecurity', 'cloud computing security', 'network architecture', 'data center security'
            ],
            
            
            'default': [
                'cybersecurity technology', 'computer security network', 'digital security system', 'information technology security', 'cyber protection'
            ]
        };
    }
    
    truncateDescription(text, maxWords = 25) {
        if (!text) return '🌱 Clique para ler a notícia completa e expandir seu conhecimento.';
        
        const words = text.split(' ');
        if (words.length <= maxWords) {
            return text;
        }
        
        return words.slice(0, maxWords).join(' ') + '...';
    }
    
    async getIntelligentImage(article, isCarousel = false) {
        const content = `${article.title} ${article.description}`.toLowerCase();
        const dimensions = isCarousel ? { width: 1200, height: 450 } : { width: 400, height: 200 };
        
        
        const articleKey = `${article.id || article.title.slice(0, 50)}`;
        
        
        if (this.cache.has(articleKey)) {
            const cachedImage = this.cache.get(articleKey);
            console.log(`📦 Usando imagem do cache para:`, article.title.slice(0, 50));
            return cachedImage;
        }
        
        
        for (const [keywords, searchTerms] of Object.entries(this.imageKeywords)) {
            if (keywords === 'default') continue;
            
            const regex = new RegExp(keywords, 'i');
            if (regex.test(content)) {
                const searchTerm = searchTerms[Math.floor(Math.random() * searchTerms.length)];
                console.log(`🔍 Buscando imagem para palavra-chave "${keywords}" com termo: "${searchTerm}"`);
                
                const imageUrl = await this.searchImageOnline(searchTerm, dimensions);
                if (imageUrl) {
                    this.cache.set(articleKey, imageUrl);
                    console.log(`🎯 Imagem encontrada e cacheada:`, imageUrl);
                    return imageUrl;
                }
            }
        }
        
        
        const defaultTerms = this.imageKeywords.default;
        const searchTerm = defaultTerms[Math.floor(Math.random() * defaultTerms.length)];
        console.log(`🎨 Buscando imagem padrão com termo: "${searchTerm}"`);
        
        const imageUrl = await this.searchImageOnline(searchTerm, dimensions);
        if (imageUrl) {
            this.cache.set(articleKey, imageUrl);
            console.log(`🎨 Imagem padrão encontrada e cacheada:`, imageUrl);
            return imageUrl;
        }
        
        
        const fallbackUrl = `https:
        this.cache.set(articleKey, fallbackUrl);
        console.log(`🆘 Usando fallback aleatório:`, fallbackUrl);
        return fallbackUrl;
    }
    
    async searchImageOnline(searchTerm, dimensions) {
        try {
            
            const imageSources = [
                () => this.searchUnsplash(searchTerm, dimensions),
                () => this.searchPexels(searchTerm, dimensions),
                () => this.searchPixabay(searchTerm, dimensions),
                () => this.generatePlaceholder(searchTerm, dimensions)
            ];
            
            
            for (const source of imageSources) {
                try {
                    const imageUrl = await source();
                    if (imageUrl && !this.usedImages.has(imageUrl)) {
                        this.usedImages.add(imageUrl);
                        return imageUrl;
                    }
                } catch (error) {
                    console.log(`⚠️ Erro em fonte de imagem:`, error.message);
                    continue;
                }
            }
            
            return null;
        } catch (error) {
            console.error(`❌ Erro na busca de imagem:`, error);
            return null;
        }
    }
    
    async searchUnsplash(searchTerm, dimensions) {
        try {
            
            const query = encodeURIComponent(searchTerm);
            const randomSeed = Math.floor(Math.random() * 10000);
            return `https:
        } catch (error) {
            console.log(`⚠️ Erro no Unsplash:`, error);
            return null;
        }
    }
    
    async searchPexels(searchTerm, dimensions) {
        try {
            
            const query = encodeURIComponent(searchTerm.replace(/\s+/g, '-'));
            const randomId = Math.floor(Math.random() * 1000) + 1;
            return `https:
        } catch (error) {
            console.log(`⚠️ Erro no Pexels:`, error);
            return null;
        }
    }
    
    async searchPixabay(searchTerm, dimensions) {
        try {
            
            const randomSeed = encodeURIComponent(searchTerm) + Math.floor(Math.random() * 10000);
            return `https:
        } catch (error) {
            console.log(`⚠️ Erro no Pixabay:`, error);
            return null;
        }
    }
    
    generatePlaceholder(searchTerm, dimensions) {
        
        const colors = {
            'hack': '2d1b69/ffffff',
            'virus': 'dc2626/ffffff', 
            'crypto': 'f59e0b/000000',
            'ai': '7c3aed/ffffff',
            'network': '059669/ffffff',
            'default': '374151/ffffff'
        };
        
        let colorScheme = colors.default;
        for (const [key, color] of Object.entries(colors)) {
            if (searchTerm.toLowerCase().includes(key)) {
                colorScheme = color;
                break;
            }
        }
        
        const text = encodeURIComponent(searchTerm.substring(0, 20));
        return `https:
    }
    
    loadIntelligentImage(imgElement) {
        try {
            const articleId = imgElement.getAttribute('data-article-id');
            if (!articleId) return;
            
            
            const article = this.findArticleById(articleId);
            if (!article) return;
            
            
            this.getIntelligentImage(article, false).then(imageUrl => {
                if (imageUrl && imgElement) {
                    imgElement.src = imageUrl;
                    imgElement.alt = `Imagem relacionada à ${article.title}`;
                    console.log(`🖼️ Imagem carregada para: ${article.title.slice(0, 50)}`);
                }
            }).catch(error => {
                console.error(`❌ Erro ao carregar imagem inteligente:`, error);
                
                imgElement.src = `https:
            });
        } catch (error) {
            console.error(`❌ Erro ao carregar imagem inteligente:`, error);
        }
    }
    
    handleImageError(imgElement, articleId) {
        
        imgElement.setAttribute('data-article-id', articleId);
        this.loadIntelligentImage(imgElement);
    }
    
    findArticleById(articleId) {
        
        const allCards = document.querySelectorAll('.news-card');
        for (const card of allCards) {
            const cardId = card.getAttribute('data-news-id');
            const title = card.querySelector('.news-title')?.textContent;
            if (cardId === articleId || title?.slice(0, 50) === articleId) {
                return {
                    id: cardId,
                    title: title,
                    description: card.querySelector('.news-description')?.textContent || ''
                };
            }
        }
        return null;
    }
    
    async getSmartImageUrl(article) {
        
            if (article.image_url && article.image_url.trim() !== '' && article.image_url !== 'null') {
            return article.image_url;
        }
        
        
        const content = `${article.title} ${article.description}`.toLowerCase();
        
        
        const imageMap = {
            'hack|attack|ataque|breach|invasion': [
                'https:
                'https:
                'https:
            ],
            'ransomware|virus|malware|trojan': [
                'https:
                'https:
                'https:
            ],
            'leak|vazamento|data|dados|breach|exposure': [
                'https:
                'https:
                'https:
            ],
            'vulnerability|vulnerabilidade|exploit|zero-day|cve': [
                'https:
                'https:
                'https:
            ],
            'phishing|scam|fraud|social engineering': [
                'https:
                'https:
                'https:
            ],
            'crypto|bitcoin|blockchain|cryptocurrency|nft': [
                'https:
                'https:
                'https:
            ],
            'ai|artificial intelligence|machine learning|neural': [
                'https:
                'https:
                'https:
            ],
            'google|microsoft|apple|meta|facebook|twitter|amazon': [
                'https:
                'https:
                'https:
            ],
            'network|rede|server|servidor|cloud|nuvem': [
                'https:
                'https:
                'https:
            ]
        };
        
        
        for (const [keywords, images] of Object.entries(imageMap)) {
            const regex = new RegExp(keywords, 'i');
            if (regex.test(content)) {
                const randomIndex = Math.floor(Math.random() * images.length);
                const selectedImage = images[randomIndex];
                
                
                const timestamp = Date.now();
                const imageWithTimestamp = selectedImage + `&t=${timestamp}`;
                
                console.log(`🎯 Imagem selecionada para "${keywords}":`, imageWithTimestamp);
                return imageWithTimestamp;
            }
        }
        
        
        const defaultImages = [
            'https:
            'https:
            'https:
            'https:
            'https:
        ];
        
        const randomDefault = Math.floor(Math.random() * defaultImages.length);
        const timestamp = Date.now();
        const defaultImage = defaultImages[randomDefault] + `&t=${timestamp}`;
        
        console.log(`🎨 Imagem padrão selecionada:`, defaultImage);
        return defaultImage;
    }
    
    getSmartCarouselImageUrl(article) {
        
        if (article.image_url) {
            return article.image_url;
        }
        
        
        const content = `${article.title} ${article.description}`.toLowerCase();
        
        
        const imageMap = {
            'hack|attack|ataque|breach|invasion': [
                'https:
                'https:
                'https:
            ],
            'ransomware|virus|malware|trojan': [
                'https:
                'https:
                'https:
            ],
            'leak|vazamento|data|dados|breach|exposure': [
                'https:
                'https:
                'https:
            ],
            'vulnerability|vulnerabilidade|exploit|zero-day|cve': [
                'https:
                'https:
                'https:
            ],
            'phishing|scam|fraud|social engineering': [
                'https:
                'https:
                'https:
            ],
            'crypto|bitcoin|blockchain|cryptocurrency|nft': [
                'https:
                'https:
                'https:
            ],
            'ai|artificial intelligence|machine learning|neural': [
                'https:
                'https:
                'https:
            ],
            'google|microsoft|apple|meta|facebook|twitter|amazon': [
                'https:
                'https:
                'https:
            ],
            'network|rede|server|servidor|cloud|nuvem': [
                'https:
                'https:
                'https:
            ]
        };
        
        
        for (const [keywords, images] of Object.entries(imageMap)) {
            const regex = new RegExp(keywords, 'i');
            if (regex.test(content)) {
                const randomIndex = Math.floor(Math.random() * images.length);
                const selectedImage = images[randomIndex];
                
                
                const timestamp = Date.now();
                const imageWithTimestamp = selectedImage + `&t=${timestamp}`;
                
                console.log(`🎯 Imagem carrossel selecionada para "${keywords}":`, imageWithTimestamp);
                return imageWithTimestamp;
            }
        }
        
        
        const defaultImages = [
            'https:
            'https:
            'https:
            'https:
            'https:
        ];
        
        const randomDefault = Math.floor(Math.random() * defaultImages.length);
        const timestamp = Date.now();
        const defaultImage = defaultImages[randomDefault] + `&t=${timestamp}`;
        
        console.log(`🎨 Imagem carrossel padrão selecionada:`, defaultImage);
        return defaultImage;
    }
}
document.addEventListener('DOMContentLoaded', function() {
    
    setTimeout(() => {
        window.newsSystem = new NewsSystem();
    }, 200);
});
function openArticleModal(articleId) {
    console.log('🔍 Abrindo modal para artigo:', articleId);
    
    
    fetch(`/backend/manual-articles.php?action=get_modal&id=${articleId}`)
        .then(response => response.text())
        .then(html => {
            
            let modal = document.getElementById('article-modal');
            if (!modal) {
                modal = document.createElement('div');
                modal.id = 'article-modal';
                modal.className = 'article-modal';
                document.body.appendChild(modal);
            }
            
            
            modal.innerHTML = html;
            modal.style.display = 'block';
            
            
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    closeArticleModal();
                }
            });
            
            
            const closeBtn = modal.querySelector('.modal-close');
            if (closeBtn) {
                closeBtn.focus();
            }
            
            console.log('✅ Modal aberto com sucesso');
        })
        .catch(error => {
            console.error('❌ Erro ao abrir modal:', error);
            alert('Erro ao carregar o artigo. Tente novamente.');
        });
}
function closeArticleModal() {
    const modal = document.getElementById('article-modal');
    if (modal) {
        modal.style.display = 'none';
        console.log('🔒 Modal fechado');
    }
}
function shareArticle(title) {
    if (navigator.share) {
        navigator.share({
            title: title,
            text: `Confira este artigo: ${title}`,
            url: window.location.href
        }).catch(error => {
            console.log('Erro ao compartilhar:', error);
            copyToClipboard(window.location.href);
        });
    } else {
        copyToClipboard(window.location.href);
    }
}
function copyToClipboard(text) {
    if (navigator.clipboard) {
        navigator.clipboard.writeText(text).then(() => {
            showNotification('Link copiado para a área de transferência!', 'success');
        }).catch(() => {
            fallbackCopyToClipboard(text);
        });
    } else {
        fallbackCopyToClipboard(text);
    }
}
function fallbackCopyToClipboard(text) {
    const textArea = document.createElement('textarea');
    textArea.value = text;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    textArea.style.top = '-999999px';
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();
    
    try {
        document.execCommand('copy');
        showNotification('Link copiado para a área de transferência!', 'success');
    } catch (err) {
        showNotification('Erro ao copiar link', 'error');
    }
    
    document.body.removeChild(textArea);
}
function showNotification(message, type = 'info') {
    
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#4a7c2a' : type === 'error' ? '#dc2626' : '#6b9a3d'};
        color: white;
        padding: 15px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        z-index: 10000;
        transform: translateX(400px);
        transition: transform 0.3s ease;
        max-width: 300px;
        font-weight: 500;
    `;
    
    notification.textContent = message;
    document.body.appendChild(notification);
    
    
    setTimeout(() => {
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    
    setTimeout(() => {
        notification.style.transform = 'translateX(400px)';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeArticleModal();
    }
});
document.addEventListener('DOMContentLoaded', function() {
    const affiliateTrack = document.getElementById('affiliate-track');
    const prevBtn = document.getElementById('affiliate-prev');
    const nextBtn = document.getElementById('affiliate-next');
    
    if (affiliateTrack && prevBtn && nextBtn) {
        let currentIndex = 0;
        
        
        function getCardWidth() {
            const card = affiliateTrack.querySelector('.affiliate-card');
            if (card) {
                const cardStyle = window.getComputedStyle(card);
                const cardWidth = card.offsetWidth;
                const gap = parseInt(window.getComputedStyle(affiliateTrack).gap) || 30;
                return cardWidth + gap;
            }
            return 330; 
        }
        
        
        function updateCarousel() {
            const cardWidth = getCardWidth();
            const translateX = -currentIndex * cardWidth;
            affiliateTrack.style.transform = `translateX(${translateX}px)`;
            
            
            prevBtn.disabled = currentIndex === 0;
            nextBtn.disabled = currentIndex >= totalCards - getVisibleCards();
        }
        
        
        function getVisibleCards() {
            const containerWidth = affiliateTrack.parentElement.offsetWidth;
            const cardWidth = getCardWidth();
            return Math.floor(containerWidth / cardWidth);
        }
        
        const totalCards = affiliateTrack.children.length;
        
        
        prevBtn.addEventListener('click', () => {
            if (currentIndex > 0) {
                currentIndex--;
                updateCarousel();
            }
        });
        
        nextBtn.addEventListener('click', () => {
            const maxIndex = totalCards - getVisibleCards();
            if (currentIndex < maxIndex) {
                currentIndex++;
                updateCarousel();
            }
        });
        
        
        window.addEventListener('resize', () => {
            
            currentIndex = 0;
            updateCarousel();
        });
        
        
        updateCarousel();
    }
});
function copyPromoCode(code) {
    if (navigator.clipboard) {
        navigator.clipboard.writeText(code).then(() => {
            showNotification(`Código promocional "${code}" copiado! 🎉`, 'success');
        }).catch(() => {
            fallbackCopyPromoCode(code);
        });
    } else {
        fallbackCopyPromoCode(code);
    }
}
function fallbackCopyPromoCode(code) {
    const textArea = document.createElement('textarea');
    textArea.value = code;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    textArea.style.top = '-999999px';
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();
    
    try {
        document.execCommand('copy');
        showNotification(`Código promocional "${code}" copiado! 🎉`, 'success');
    } catch (err) {
        showNotification('Erro ao copiar código promocional', 'error');
    }
    
    document.body.removeChild(textArea);
}