(function() {
    'use strict';
    
    
    function initLazyLoading() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        if (img.dataset.src) {
                            img.src = img.dataset.src;
                            img.removeAttribute('data-src');
                            img.classList.remove('lazy');
                            observer.unobserve(img);
                        }
                    }
                });
            }, { rootMargin: '50px 0px' });
            document.querySelectorAll('img[data-src]').forEach(img => {
                imageObserver.observe(img);
            });
        }
    }
    
    function preloadCritical() {
        const critical = [
            '/css/critical.css',
            '/styles.css'
        ];
        
        critical.forEach(url => {
            const link = document.createElement('link');
            link.rel = 'preload';
            link.as = 'style';
            link.href = url;
            document.head.appendChild(link);
        });
    }
    
    function optimizeAnimations() {
        const prefersReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)').matches;
        if (prefersReducedMotion) {
            document.documentElement.style.setProperty('--animation-duration', '0s');
        }
        
        
        document.addEventListener('visibilitychange', () => {
            const animatedElements = document.querySelectorAll('[class*="animate"], [class*="float"], [class*="pulse"]');
            animatedElements.forEach(el => {
                el.style.animationPlayState = document.hidden ? 'paused' : 'running';
            });
        });
    }
    
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    function optimizeScroll() {
        let ticking = false;
        
        const updateScrollElements = debounce(() => {
            const scrollTop = window.pageYOffset;
            const backToTop = document.querySelector('.back-to-top');
            
            if (backToTop) {
                if (scrollTop > 300) {
                    backToTop.classList.add('show');
                } else {
                    backToTop.classList.remove('show');
                }
            }
            
            ticking = false;
        }, 16);
        window.addEventListener('scroll', () => {
            if (!ticking) {
                requestAnimationFrame(updateScrollElements);
                ticking = true;
            }
        }, { passive: true });
    }
    
    function loadNonCriticalCSS() {
        const nonCritical = [
            '/css/news-psychedelic.css',
            '/css/sections.css',
            '/css/store.css'
        ];
        
        nonCritical.forEach((url, index) => {
            setTimeout(() => {
                const link = document.createElement('link');
                link.rel = 'stylesheet';
                link.href = url;
                link.media = 'print';
                link.onload = function() { this.media = 'all'; };
                document.head.appendChild(link);
            }, index * 100);
        });
    }
    
    function registerServiceWorker() {
        // Temporariamente desabilitado para debug
        // if ('serviceWorker' in navigator) {
        //     navigator.serviceWorker.register('/sw.js').catch(() => {
        //         
        //     });
        // }
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    function init() {
        initLazyLoading();
        optimizeAnimations();
        optimizeScroll();
        
        
        window.addEventListener('load', () => {
            setTimeout(() => {
                loadNonCriticalCSS();
                registerServiceWorker();
            }, 100);
        });
    }
})();
