const PRODUCTS = {
  'scanner-cli': {
    name: 'Scanner de Vulnerabilidades (CLI)',
    price: 8900, 
    priceBRL: 89.00,
    description: 'Varredura rápida de portas, banners e CVEs conhecidas. Relatório em JSON/HTML.',
    type: 'product'
  },
  'pentest-bot': {
    name: 'Bot de Pentest Automatizado',
    price: 14900,
    priceBRL: 149.00,
    description: 'Execução automatizada de checks OWASP Top 10 com export PDF.',
    type: 'product'
  },
  'forense-kit': {
    name: 'Kit de Scripts de Forense',
    price: 11900,
    priceBRL: 119.00,
    description: 'Coleta de artefatos, timeline e hashing verificado. Pronto para laudos.',
    type: 'product'
  },
  'phishing-api': {
    name: 'Detector de Phishing (API)',
    price: 9900,
    priceBRL: 99.00,
    description: 'Classificador simples para URLs suspeitas. Retorno JSON com score.',
    type: 'product'
  }
};
const SERVICES = {
  'siem-config': {
    name: 'Configuração de SIEM',
    price: 250000, 
    priceBRL: 2500.00,
    description: 'Implementação e configuração completa de plataformas SIEM (Splunk, Elastic Stack, QRadar)',
    type: 'service',
    deliveryTime: '7-14 dias úteis',
    includes: [
      'Instalação e configuração inicial',
      'Criação de dashboards personalizados',
      'Configuração de alertas e regras',
      'Treinamento da equipe (4h)',
      '30 dias de suporte incluso'
    ]
  },
  'firewall-setup': {
    name: 'Setup de Firewall Corporativo',
    price: 180000, 
    priceBRL: 1800.00,
    description: 'Configuração profissional de firewalls (pfSense, FortiGate, SonicWall)',
    type: 'service',
    deliveryTime: '5-10 dias úteis',
    includes: [
      'Análise de infraestrutura atual',
      'Configuração de regras de segurança',
      'VPN site-to-site e client',
      'Monitoramento e logs',
      'Documentação técnica completa'
    ]
  },
  'vuln-mgmt-setup': {
    name: 'Plataforma de Vulnerability Management',
    price: 320000, 
    priceBRL: 3200.00,
    description: 'Implementação de soluções de gestão de vulnerabilidades (Nessus, OpenVAS, Rapid7)',
    type: 'service',
    deliveryTime: '10-15 dias úteis',
    includes: [
      'Instalação e configuração da plataforma',
      'Criação de políticas de scan',
      'Integração com ferramentas existentes',
      'Relatórios automatizados',
      'Workflow de correção de vulnerabilidades'
    ]
  },
  'soc-setup': {
    name: 'SOC as a Service - Setup',
    price: 550000, 
    priceBRL: 5500.00,
    description: 'Implementação completa de Centro de Operações de Segurança (SOC)',
    type: 'service',
    deliveryTime: '15-30 dias úteis',
    includes: [
      'Arquitetura e design do SOC',
      'Configuração de ferramentas integradas',
      'Playbooks de resposta a incidentes',
      'Treinamento da equipe SOC (8h)',
      '60 dias de suporte especializado'
    ]
  },
  'cloud-migration': {
    name: 'Migração Segura para Cloud',
    price: 420000, 
    priceBRL: 4200.00,
    description: 'Consultoria especializada para migração segura para AWS, Azure ou GCP',
    type: 'service',
    deliveryTime: '20-30 dias úteis',
    includes: [
      'Análise de segurança atual',
      'Planejamento de migração',
      'Configuração de segurança na cloud',
      'Compliance e governança',
      'Monitoramento e alertas'
    ]
  },
  'custom-training': {
    name: 'Treinamento Personalizado',
    price: 120000, 
    priceBRL: 1200.00,
    description: 'Treinamentos customizados em cibersegurança para sua equipe',
    type: 'service',
    unit: '/dia',
    deliveryTime: 'Agendamento conforme disponibilidade',
    includes: [
      'Análise de necessidades da equipe',
      'Conteúdo personalizado',
      'Simulações práticas',
      'Material didático incluso',
      'Certificado de participação'
    ]
  }
};
let stripe = null;
let payramInitialized = false;
document.addEventListener('DOMContentLoaded', function() {
  
  initializeStripe();
  
  
  initializePayRam();
  
  
  setupBuyButtons();
});
function initializeStripe() {
  try {
    // Verificar se o Stripe carregou
    if (typeof window.Stripe !== 'undefined') {
      stripe = window.Stripe('pk_test_51234567890abcdef');
      console.log('✅ Stripe inicializado com sucesso');
    } else {
      console.warn('⚠️ Stripe não carregou - funcionará sem cartão de crédito');
      stripe = null;
    }
  } catch (error) {
    console.error('❌ Erro ao inicializar Stripe:', error);
    stripe = null;
  }
}
function initializePayRam() {
  try {
    
    
    window.payramConfig = {
      apiKey: 'YOUR_PAYRAM_API_KEY', 
      environment: 'sandbox', 
      supportedCryptos: ['BTC', 'ETH', 'USDT', 'PIX']
    };
    payramInitialized = true;
    console.log('✅ PayRam configurado com sucesso');
  } catch (error) {
    console.error('❌ Erro ao configurar PayRam:', error);
  }
}
function setupBuyButtons() {
  const buyButtons = document.querySelectorAll('.buy-button, .service-btn');
  console.log('🛒 Configurando botões de compra:', buyButtons.length);
  
  buyButtons.forEach(btn => {
    btn.addEventListener('click', (e) => {
      console.log('🛒 Botão clicado:', btn, 'ProductId:', btn.getAttribute('data-product-id'));
      e.preventDefault();
      e.stopPropagation(); // Importante para não conflitar com outros listeners
      const productId = btn.getAttribute('data-product-id');
      
      
      let item = PRODUCTS[productId] || SERVICES[productId];
      
      if (item) {
        // TODOS os itens (produtos E serviços) devem abrir modal de pagamento
        console.log('📦 Abrindo modal de pagamento para:', item.type, productId);
        showPaymentModal(productId, item);
      } else {
        console.error('Item não encontrado:', productId);
      }
    });
  });
}
function showServiceModal(serviceId, service) {
  
  const modal = document.createElement('div');
  modal.className = 'payment-modal service-modal';
  modal.innerHTML = `
    <div class="modal-overlay">
      <div class="modal-content">
        <div class="modal-header">
          <h3>🔧 ${service.name}</h3>
          <button class="close-modal" aria-label="Fechar">&times;</button>
        </div>
        
        <div class="product-summary">
          <h4>${service.name}</h4>
          <p>${service.description}</p>
          
          <div class="service-details">
            <div class="detail-item">
              <strong>⏱️ Prazo de Entrega:</strong> ${service.deliveryTime}
            </div>
            
            <div class="detail-item">
              <strong>📋 O que está incluído:</strong>
              <ul class="includes-list">
                ${service.includes.map(item => `<li>✅ ${item}</li>`).join('')}
              </ul>
            </div>
          </div>
          
          <div class="price">
            R$ ${service.priceBRL.toLocaleString('pt-BR', { minimumFractionDigits: 2 })}${service.unit || ''}
          </div>
        </div>
        
        <div class="payment-options">
          <div class="service-notice">
            <p>🤝 <strong>Processo de Contratação:</strong></p>
            <p>1. Clique em "Solicitar Orçamento"</p>
            <p>2. Enviaremos um formulário detalhado</p>
            <p>3. Análise técnica e proposta personalizada</p>
            <p>4. Pagamento após aprovação da proposta</p>
          </div>
          
          <button class="payment-btn service-quote-btn" data-service="${serviceId}">
            <i class="fas fa-envelope"></i>
            <span>Solicitar Orçamento Personalizado</span>
            <small>Resposta em até 24h</small>
          </button>
          
          <button class="payment-btn service-whatsapp-btn" data-service="${serviceId}">
            <i class="fab fa-whatsapp"></i>
            <span>Falar no WhatsApp</span>
            <small>Atendimento direto</small>
          </button>
        </div>
      </div>
    </div>
  `;
  
  document.body.appendChild(modal);
  
  
  modal.querySelector('.close-modal').addEventListener('click', () => {
    document.body.removeChild(modal);
  });
  
  modal.querySelector('.modal-overlay').addEventListener('click', (e) => {
    if (e.target === modal.querySelector('.modal-overlay')) {
      document.body.removeChild(modal);
    }
  });
  
  
  modal.querySelector('.service-quote-btn').addEventListener('click', () => {
    requestServiceQuote(serviceId, service);
    document.body.removeChild(modal);
  });
  
  modal.querySelector('.service-whatsapp-btn').addEventListener('click', () => {
    contactWhatsApp(serviceId, service);
    document.body.removeChild(modal);
  });
}
function requestServiceQuote(serviceId, service) {
  
  const subject = encodeURIComponent(`Orçamento - ${service.name}`);
  const body = encodeURIComponent(`
Olá!
Gostaria de solicitar um orçamento para o serviço:
🔧 Serviço: ${service.name}
💰 Valor base: R$ ${service.priceBRL.toLocaleString('pt-BR', { minimumFractionDigits: 2 })}${service.unit || ''}
⏱️ Prazo: ${service.deliveryTime}
📋 Descrição: ${service.description}
Por favor, enviem mais informações sobre:
- Processo de implementação
- Cronograma detalhado  
- Requisitos técnicos
- Forma de pagamento
Aguardo retorno!
Atenciosamente.
  `);
  
  window.location.href = `mailto:mcbfk1981@gmail.com?subject=${subject}&body=${body}`;
}
function contactWhatsApp(serviceId, service) {
  const message = encodeURIComponent(`
Olá! 👋
Tenho interesse no serviço: *${service.name}*
Valor: R$ ${service.priceBRL.toLocaleString('pt-BR', { minimumFractionDigits: 2 })}${service.unit || ''}
Gostaria de mais informações sobre o processo de contratação.
  `);
  
  window.open(`https://wa.me/5511999999999?text=${message}`, '_blank');
}
function showPaymentModal(productId, product) {
  console.log('💳 Abrindo modal de pagamento para:', productId, product);
  
  // Remove any existing payment modals
  const existingModal = document.querySelector('.payment-modal');
  if (existingModal) {
    existingModal.remove();
  }
  
  const modal = document.createElement('div');
  modal.className = 'payment-modal';
  modal.style.cssText = `
    position: fixed !important;
    top: 0 !important;
    left: 0 !important;
    width: 100% !important;
    height: 100% !important;
    z-index: 99999 !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    opacity: 1 !important;
    visibility: visible !important;
    pointer-events: auto !important;
    background: 
      radial-gradient(ellipse at top, rgba(45, 80, 22, 0.4), transparent),
      radial-gradient(ellipse at bottom, rgba(13, 30, 17, 0.6), transparent),
      linear-gradient(135deg, rgba(15, 15, 35, 0.9), rgba(26, 26, 46, 0.95)) !important;
    backdrop-filter: blur(15px) !important;
    animation: modalBackgroundShift 8s ease-in-out infinite !important;
  `;
  modal.innerHTML = `
      <div class="modal-content" style="
        position: relative;
        background: 
          radial-gradient(ellipse at top left, rgba(144, 238, 144, 0.1), transparent),
          radial-gradient(ellipse at top right, rgba(74, 124, 42, 0.15), transparent),
          radial-gradient(ellipse at bottom left, rgba(107, 154, 61, 0.1), transparent),
          radial-gradient(ellipse at bottom right, rgba(143, 179, 102, 0.12), transparent),
          linear-gradient(135deg, #0a1a0e, #152318, #0d0d20, #1a1a2e);
        border-radius: 30px;
        padding: 45px;
        max-width: 600px;
        width: 95%;
        max-height: 90vh;
        overflow-y: auto;
        scrollbar-width: thin;
        scrollbar-color: rgba(144, 238, 144, 0.6) rgba(144, 238, 144, 0.1);
        border: 2px solid transparent;
        background-clip: padding-box;
        box-shadow: 
          0 0 50px rgba(144, 238, 144, 0.3),
          0 0 100px rgba(144, 238, 144, 0.15),
          0 20px 40px rgba(0, 0, 0, 0.4),
          inset 0 1px 0 rgba(144, 238, 144, 0.2),
          inset 0 -1px 0 rgba(144, 238, 144, 0.1);
        animation: 
          futuristicGlow 3s ease-in-out infinite alternate,
          modalFloat 6s ease-in-out infinite;
        backdrop-filter: blur(20px);
        overflow: hidden;
      ">
        
        <!-- Campo de estrelas espacial estilo Grok -->
        <div style="
          position: absolute;
          top: 0;
          left: 0;
          right: 0;
          bottom: 0;
          background-image: 
            radial-gradient(1px 1px at 80px 40px, #90ee90, transparent),
            radial-gradient(1px 1px at 180px 90px, #b9f9b9, transparent),
            radial-gradient(2px 2px at 120px 160px, #90ee90, transparent),
            radial-gradient(1px 1px at 320px 80px, #4a7c23, transparent),
            radial-gradient(1px 1px at 220px 220px, #90ee90, transparent),
            radial-gradient(2px 2px at 400px 140px, #b9f9b9, transparent),
            radial-gradient(1px 1px at 60px 180px, #4a7c23, transparent),
            radial-gradient(1px 1px at 340px 200px, #90ee90, transparent),
            radial-gradient(2px 2px at 160px 60px, #b9f9b9, transparent),
            radial-gradient(1px 1px at 280px 160px, #4a7c23, transparent),
            radial-gradient(1px 1px at 420px 240px, #90ee90, transparent),
            radial-gradient(2px 2px at 100px 280px, #b9f9b9, transparent),
            radial-gradient(1px 1px at 260px 40px, #4a7c23, transparent),
            radial-gradient(1px 1px at 380px 180px, #90ee90, transparent),
            radial-gradient(2px 2px at 140px 320px, #b9f9b9, transparent);
          animation: starField 20s linear infinite;
          pointer-events: none;
          border-radius: 30px;
          z-index: 0;
        "></div>
        
        <!-- Nebulosa canábica de fundo -->
        <div style="
          position: absolute;
          top: 0;
          left: 0;
          right: 0;
          bottom: 0;
          background: 
            radial-gradient(ellipse 300px 200px at 20% 30%, rgba(45, 80, 22, 0.3), transparent),
            radial-gradient(ellipse 250px 150px at 80% 70%, rgba(74, 124, 42, 0.2), transparent),
            radial-gradient(ellipse 200px 300px at 60% 20%, rgba(107, 154, 61, 0.25), transparent);
          animation: nebulaShift 25s ease-in-out infinite;
          pointer-events: none;
          border-radius: 30px;
          z-index: 1;
        "></div>
        
        <!-- Linhas de circuito futuristas -->
        <div style="
          position: absolute;
          top: 0;
          left: 0;
          right: 0;
          bottom: 0;
          background: 
            linear-gradient(90deg, transparent 48%, rgba(144, 238, 144, 0.1) 49%, rgba(144, 238, 144, 0.1) 51%, transparent 52%),
            linear-gradient(0deg, transparent 48%, rgba(144, 238, 144, 0.08) 49%, rgba(144, 238, 144, 0.08) 51%, transparent 52%);
          animation: circuitPulse 4s ease-in-out infinite;
          pointer-events: none;
          border-radius: 30px;
          z-index: 1;
        "></div>
        
        <!-- Conteúdo principal -->
        <div style="position: relative; z-index: 10;">
        <div class="modal-header" style="
          display: flex;
          justify-content: space-between;
          align-items: center;
          margin-bottom: 30px;
          padding: 20px;
          border: 2px solid #90ee90;
          border-radius: 20px;
          background: linear-gradient(135deg, rgba(144, 238, 144, 0.05), rgba(144, 238, 144, 0.1));
          position: relative;
        ">
          <div class="modal-title-container" style="display: flex; align-items: center; gap: 20px;">
            <div class="modal-icon" style="
              font-size: 2.5rem; 
              background: radial-gradient(circle, rgba(144, 238, 144, 0.2), transparent);
              padding: 10px;
              border-radius: 50%;
              border: 2px solid rgba(144, 238, 144, 0.3);
            ">💳</div>
            <div>
              <h3 style="
                color: #90ee90; 
                margin: 0; 
                font-size: 1.8rem;
                text-shadow: 0 0 10px rgba(144, 238, 144, 0.5);
                font-weight: bold;
              ">Pagamento Seguro</h3>
              <p style="
                color: #b9f9b9; 
                margin: 5px 0 0 0; 
                font-size: 0.9rem;
                opacity: 0.8;
              ">Escolha sua forma de pagamento preferida</p>
            </div>
          </div>
          <button class="close-modal" aria-label="Fechar" style="
            background: linear-gradient(135deg, rgba(144, 238, 144, 0.1), rgba(144, 238, 144, 0.2));
            border: 2px solid rgba(144, 238, 144, 0.3);
            color: #90ee90;
            font-size: 1.5rem;
            cursor: pointer;
            padding: 8px;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            position: relative;
            overflow: hidden;
            text-shadow: 0 0 8px rgba(144, 238, 144, 0.6);
            box-shadow: 0 4px 15px rgba(144, 238, 144, 0.2);
          ">&times;</button>
        </div>
        
        <div class="product-summary" style="
          text-align: center;
          margin-bottom: 30px;
          padding: 25px;
          background: linear-gradient(135deg, rgba(144, 238, 144, 0.05), rgba(144, 238, 144, 0.15));
          border-radius: 20px;
          border: 2px solid rgba(144, 238, 144, 0.3);
          box-shadow: 
            0 0 20px rgba(144, 238, 144, 0.1),
            inset 0 1px 0 rgba(144, 238, 144, 0.1);
          position: relative;
        ">
          <div style="
            position: absolute;
            top: -1px;
            left: -1px;
            right: -1px;
            height: 2px;
            background: linear-gradient(90deg, transparent, #90ee90, transparent);
            border-radius: 20px 20px 0 0;
          "></div>
          <h4 style="
            color: #90ee90; 
            margin: 0 0 15px 0; 
            font-size: 1.5rem;
            text-shadow: 0 0 15px rgba(144, 238, 144, 0.6);
            font-weight: bold;
          ">${product.name}</h4>
          <p style="
            color: #b9f9b9; 
            margin: 0 0 20px 0; 
            opacity: 0.9;
            line-height: 1.4;
            font-size: 1rem;
          ">${product.description}</p>
          <div class="price" style="
            color: #90ee90; 
            font-size: 2.2rem; 
            font-weight: bold;
            text-shadow: 0 0 20px rgba(144, 238, 144, 0.8);
            background: linear-gradient(135deg, #90ee90, #b9f9b9);
            -webkit-background-clip: text;
            background-clip: text;
            -webkit-text-fill-color: transparent;
          ">R$ ${product.priceBRL.toFixed(2)}</div>
        </div>
        
        <div class="payment-options" style="
          display: grid; 
          grid-template-columns: 1fr 1fr 1fr; 
          gap: 20px; 
          margin-top: 40px;
        ">
          <button class="payment-btn stripe-btn" data-method="stripe" style="
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 12px;
            padding: 20px 15px;
            background: 
              radial-gradient(ellipse at center, rgba(144, 238, 144, 0.15), transparent),
              linear-gradient(135deg, rgba(15, 15, 35, 0.8), rgba(26, 26, 46, 0.9));
            border: 1px solid rgba(144, 238, 144, 0.6);
            border-radius: 20px;
            color: #90ee90;
            cursor: pointer;
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            font-size: 0.9rem;
            position: relative;
            overflow: hidden;
            backdrop-filter: blur(10px);
            min-height: 120px;
            justify-content: center;
          ">
            <!-- Estrelas de fundo no botão -->
            <div style="
              position: absolute;
              top: 0;
              left: 0;
              right: 0;
              bottom: 0;
              background-image: 
                radial-gradient(1px 1px at 20% 30%, rgba(144, 238, 144, 0.8), transparent),
                radial-gradient(1px 1px at 80% 20%, rgba(144, 238, 144, 0.6), transparent),
                radial-gradient(1px 1px at 60% 80%, rgba(144, 238, 144, 0.7), transparent);
              animation: buttonStars 8s linear infinite;
              pointer-events: none;
              border-radius: 20px;
            "></div>
            <div class="payment-icon" style="
              font-size: 2rem;
              background: radial-gradient(circle, rgba(144, 238, 144, 0.3), rgba(144, 238, 144, 0.1));
              padding: 8px;
              border-radius: 50%;
              border: 1px solid rgba(144, 238, 144, 0.5);
              position: relative;
              z-index: 2;
            ">💳</div>
            <div class="payment-info" style="text-align: center; position: relative; z-index: 2;">
              <span style="
                display: block; 
                font-weight: bold; 
                font-size: 1rem;
                text-shadow: 0 0 8px rgba(144, 238, 144, 0.6);
                margin-bottom: 4px;
              ">Cartão</span>
              <small style="opacity: 0.9; color: #b9f9b9; font-size: 0.8rem;">Stripe</small>
            </div>
          </button>
          
          <button class="payment-btn payram-btn" data-method="payram" style="
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 12px;
            padding: 20px 15px;
            background: 
              radial-gradient(ellipse at center, rgba(144, 238, 144, 0.12), transparent),
              linear-gradient(135deg, rgba(15, 15, 35, 0.8), rgba(26, 26, 46, 0.9));
            border: 1px solid rgba(144, 238, 144, 0.5);
            border-radius: 20px;
            color: #90ee90;
            cursor: pointer;
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            font-size: 0.9rem;
            position: relative;
            overflow: hidden;
            backdrop-filter: blur(10px);
            min-height: 120px;
            justify-content: center;
          ">
            <!-- Estrelas de fundo no botão -->
            <div style="
              position: absolute;
              top: 0;
              left: 0;
              right: 0;
              bottom: 0;
              background-image: 
                radial-gradient(1px 1px at 15% 25%, rgba(144, 238, 144, 0.7), transparent),
                radial-gradient(1px 1px at 85% 15%, rgba(144, 238, 144, 0.5), transparent),
                radial-gradient(1px 1px at 55% 85%, rgba(144, 238, 144, 0.8), transparent);
              animation: buttonStars 8s linear infinite 0.5s;
              pointer-events: none;
              border-radius: 20px;
            "></div>
            <div class="payment-icon" style="
              font-size: 2rem;
              background: radial-gradient(circle, rgba(144, 238, 144, 0.25), rgba(144, 238, 144, 0.08));
              padding: 8px;
              border-radius: 50%;
              border: 1px solid rgba(144, 238, 144, 0.4);
              position: relative;
              z-index: 2;
            ">₿</div>
            <div class="payment-info" style="text-align: center; position: relative; z-index: 2;">
              <span style="
                display: block; 
                font-weight: bold; 
                font-size: 1rem;
                text-shadow: 0 0 8px rgba(144, 238, 144, 0.6);
                margin-bottom: 4px;
              ">Crypto</span>
              <small style="opacity: 0.9; color: #b9f9b9; font-size: 0.8rem;">Bitcoin</small>
            </div>
          </button>
          
          <button class="payment-btn pix-btn" data-method="pix" style="
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 12px;
            padding: 20px 15px;
            background: 
              radial-gradient(ellipse at center, rgba(144, 238, 144, 0.18), transparent),
              linear-gradient(135deg, rgba(15, 15, 35, 0.8), rgba(26, 26, 46, 0.9));
            border: 1px solid rgba(144, 238, 144, 0.7);
            border-radius: 20px;
            color: #90ee90;
            cursor: pointer;
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            font-size: 0.9rem;
            position: relative;
            overflow: hidden;
            backdrop-filter: blur(10px);
            min-height: 120px;
            justify-content: center;
            box-shadow: 0 0 10px rgba(144, 238, 144, 0.15);
          ">
            <!-- Estrelas de fundo no botão -->
            <div style="
              position: absolute;
              top: 0;
              left: 0;
              right: 0;
              bottom: 0;
              background-image: 
                radial-gradient(1px 1px at 25% 35%, rgba(144, 238, 144, 0.9), transparent),
                radial-gradient(1px 1px at 75% 25%, rgba(144, 238, 144, 0.7), transparent),
                radial-gradient(1px 1px at 45% 75%, rgba(144, 238, 144, 0.8), transparent);
              animation: buttonStars 8s linear infinite 1s;
              pointer-events: none;
              border-radius: 20px;
            "></div>
            <div class="payment-icon" style="
              font-size: 2rem;
              background: radial-gradient(circle, rgba(144, 238, 144, 0.35), rgba(144, 238, 144, 0.12));
              padding: 8px;
              border-radius: 50%;
              border: 1px solid rgba(144, 238, 144, 0.6);
              position: relative;
              z-index: 2;
            ">📱</div>
            <div class="payment-info" style="text-align: center; position: relative; z-index: 2;">
              <span style="
                display: block; 
                font-weight: bold; 
                font-size: 1rem;
                text-shadow: 0 0 8px rgba(144, 238, 144, 0.6);
                margin-bottom: 4px;
              ">PIX</span>
              <small style="opacity: 0.9; color: #b9f9b9; font-size: 0.8rem;">Instantâneo</small>
            </div>
          </button>
        </div>
        
        <!-- Separador -->
        <div style="
          margin: 30px 0;
          display: flex;
          align-items: center;
          gap: 15px;
        ">
          <div style="flex: 1; height: 1px; background: linear-gradient(90deg, transparent, rgba(144, 238, 144, 0.5), transparent);"></div>
          <span style="color: #90ee90; font-size: 0.9rem; opacity: 0.8;">ou</span>
          <div style="flex: 1; height: 1px; background: linear-gradient(90deg, transparent, rgba(144, 238, 144, 0.5), transparent);"></div>
        </div>
        
        <!-- Botão de Orçamento Profissional -->
        <button class="professional-quote-btn" data-action="professional-quote" style="
          width: 100%;
          display: flex;
          align-items: center;
          justify-content: center;
          gap: 15px;
          padding: 20px;
          background: 
            radial-gradient(ellipse at center, rgba(144, 238, 144, 0.08), transparent),
            linear-gradient(135deg, rgba(15, 15, 35, 0.9), rgba(26, 26, 46, 0.95));
          border: 2px dashed rgba(144, 238, 144, 0.6);
          border-radius: 25px;
          color: #90ee90;
          cursor: pointer;
          transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
          font-size: 1rem;
          position: relative;
          overflow: hidden;
          backdrop-filter: blur(10px);
        ">
          <!-- Estrelas de fundo -->
          <div style="
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: 
              radial-gradient(1px 1px at 20% 40%, rgba(144, 238, 144, 0.6), transparent),
              radial-gradient(1px 1px at 80% 30%, rgba(144, 238, 144, 0.4), transparent),
              radial-gradient(1px 1px at 60% 70%, rgba(144, 238, 144, 0.5), transparent);
            animation: buttonStars 10s linear infinite;
            pointer-events: none;
            border-radius: 25px;
          "></div>
          <div style="
            font-size: 2rem;
            background: radial-gradient(circle, rgba(144, 238, 144, 0.2), rgba(144, 238, 144, 0.05));
            padding: 10px;
            border-radius: 50%;
            border: 1px solid rgba(144, 238, 144, 0.4);
            position: relative;
            z-index: 2;
          ">📋</div>
          <div style="position: relative; z-index: 2; text-align: left;">
            <span style="
              display: block; 
              font-weight: bold; 
              font-size: 1.1rem;
              text-shadow: 0 0 10px rgba(144, 238, 144, 0.6);
              margin-bottom: 4px;
            ">Orçamento Profissional</span>
            <small style="opacity: 0.9; color: #b9f9b9; font-size: 0.85rem;">Receba proposta detalhada por email</small>
          </div>
        </button>
        </div>
      </div>
      
      <style>
        /* Scroll personalizado para WebKit */
        .modal-content::-webkit-scrollbar {
          width: 8px;
        }
        .modal-content::-webkit-scrollbar-track {
          background: rgba(144, 238, 144, 0.1);
          border-radius: 10px;
        }
        .modal-content::-webkit-scrollbar-thumb {
          background: linear-gradient(135deg, rgba(144, 238, 144, 0.6), rgba(144, 238, 144, 0.8));
          border-radius: 10px;
          box-shadow: 0 0 10px rgba(144, 238, 144, 0.4);
        }
        .modal-content::-webkit-scrollbar-thumb:hover {
          background: linear-gradient(135deg, rgba(144, 238, 144, 0.8), rgba(144, 238, 144, 1));
        }
        
        /* Animações futuristas */
        @keyframes modalBackgroundShift {
          0%, 100% { 
            background: 
              radial-gradient(ellipse at top, rgba(45, 80, 22, 0.4), transparent),
              radial-gradient(ellipse at bottom, rgba(13, 30, 17, 0.6), transparent),
              linear-gradient(135deg, rgba(15, 15, 35, 0.9), rgba(26, 26, 46, 0.95));
          }
          50% { 
            background: 
              radial-gradient(ellipse at bottom, rgba(45, 80, 22, 0.5), transparent),
              radial-gradient(ellipse at top, rgba(13, 30, 17, 0.7), transparent),
              linear-gradient(135deg, rgba(26, 26, 46, 0.95), rgba(15, 15, 35, 0.9));
          }
        }
        
        @keyframes futuristicGlow {
          0% { 
            box-shadow: 
              0 0 50px rgba(144, 238, 144, 0.3),
              0 0 100px rgba(144, 238, 144, 0.15),
              0 20px 40px rgba(0, 0, 0, 0.4),
              inset 0 1px 0 rgba(144, 238, 144, 0.2);
          }
          100% { 
            box-shadow: 
              0 0 70px rgba(144, 238, 144, 0.5),
              0 0 140px rgba(144, 238, 144, 0.25),
              0 30px 60px rgba(0, 0, 0, 0.6),
              inset 0 1px 0 rgba(144, 238, 144, 0.4);
          }
        }
        
        @keyframes modalFloat {
          0%, 100% { transform: translateY(0px) rotate(0deg); }
          25% { transform: translateY(-2px) rotate(0.2deg); }
          50% { transform: translateY(-4px) rotate(0deg); }
          75% { transform: translateY(-2px) rotate(-0.2deg); }
        }
        
        @keyframes starField {
          0% { transform: translateY(0px) translateX(0px) rotate(0deg); }
          25% { transform: translateY(-2px) translateX(1px) rotate(0.5deg); }
          50% { transform: translateY(-1px) translateX(-1px) rotate(0deg); }
          75% { transform: translateY(-3px) translateX(2px) rotate(-0.5deg); }
          100% { transform: translateY(0px) translateX(0px) rotate(0deg); }
        }
        
        @keyframes nebulaShift {
          0%, 100% { 
            background: 
              radial-gradient(ellipse 300px 200px at 20% 30%, rgba(45, 80, 22, 0.3), transparent),
              radial-gradient(ellipse 250px 150px at 80% 70%, rgba(74, 124, 42, 0.2), transparent),
              radial-gradient(ellipse 200px 300px at 60% 20%, rgba(107, 154, 61, 0.25), transparent);
          }
          33% { 
            background: 
              radial-gradient(ellipse 250px 180px at 70% 40%, rgba(45, 80, 22, 0.25), transparent),
              radial-gradient(ellipse 300px 120px at 30% 80%, rgba(74, 124, 42, 0.3), transparent),
              radial-gradient(ellipse 180px 250px at 80% 60%, rgba(107, 154, 61, 0.2), transparent);
          }
          66% { 
            background: 
              radial-gradient(ellipse 280px 160px at 40% 70%, rgba(45, 80, 22, 0.35), transparent),
              radial-gradient(ellipse 200px 200px at 60% 30%, rgba(74, 124, 42, 0.25), transparent),
              radial-gradient(ellipse 220px 280px at 20% 80%, rgba(107, 154, 61, 0.3), transparent);
          }
        }
        
        @keyframes buttonStars {
          0% { opacity: 0.6; transform: scale(1) rotate(0deg); }
          25% { opacity: 1; transform: scale(1.1) rotate(90deg); }
          50% { opacity: 0.8; transform: scale(1) rotate(180deg); }
          75% { opacity: 1; transform: scale(1.05) rotate(270deg); }
          100% { opacity: 0.6; transform: scale(1) rotate(360deg); }
        }
        
        @keyframes circuitPulse {
          0%, 100% { opacity: 0.3; }
          50% { opacity: 0.6; }
        }
        
        @keyframes shimmer {
          0% { left: -100%; }
          100% { left: 100%; }
        }
        
        .payment-btn:hover {
          transform: translateY(-8px) scale(1.05) rotateY(5deg) !important;
          box-shadow: 
            0 20px 40px rgba(144, 238, 144, 0.5),
            0 0 30px rgba(144, 238, 144, 0.4),
            inset 0 0 20px rgba(144, 238, 144, 0.1) !important;
          border-color: rgba(144, 238, 144, 1) !important;
          backdrop-filter: blur(15px) !important;
        }
        
        .payment-btn:active {
          transform: translateY(-4px) scale(1.02) !important;
          box-shadow: 
            0 10px 20px rgba(144, 238, 144, 0.6),
            0 0 15px rgba(144, 238, 144, 0.5) !important;
        }
        
        .close-modal:hover {
          background: linear-gradient(135deg, rgba(144, 238, 144, 0.3), rgba(144, 238, 144, 0.5)) !important;
          transform: rotate(90deg) scale(1.1) !important;
          box-shadow: 0 0 20px rgba(144, 238, 144, 0.6) !important;
        }
      </style>
  `;
  
  document.body.appendChild(modal);
  console.log('✅ Modal de pagamento adicionado ao DOM e deve estar visível!');
  
  // Debug: verificar se modal está visível
  setTimeout(() => {
    const modalRect = modal.getBoundingClientRect();
    const computedStyle = window.getComputedStyle(modal);
    console.log('🔍 Debug Modal:', {
      'Posição': modalRect,
      'Display': computedStyle.display,
      'Opacity': computedStyle.opacity,
      'Visibility': computedStyle.visibility,
      'Z-Index': computedStyle.zIndex,
      'Background': computedStyle.background
    });
    
    // Modal está funcionando perfeitamente! 🎉
  }, 100);
  
  
  modal.querySelector('.close-modal').addEventListener('click', () => {
    document.body.removeChild(modal);
  });
  
  // Clicar fora do conteúdo fecha o modal
  modal.addEventListener('click', (e) => {
    if (e.target === modal) {
      document.body.removeChild(modal);
    }
  });
  
  
  modal.querySelectorAll('.payment-btn').forEach(btn => {
    btn.addEventListener('click', () => {
      const method = btn.getAttribute('data-method');
      document.body.removeChild(modal);
      
      switch(method) {
        case 'stripe':
          processStripePayment(productId, product);
          break;
        case 'payram':
          processPayRamPayment(productId, product);
          break;
        case 'pix':
          processPixPayment(productId, product);
          break;
      }
    });
  });
  
  // Event listener para orçamento profissional
  modal.querySelector('.professional-quote-btn').addEventListener('click', () => {
    document.body.removeChild(modal);
    openProfessionalQuoteModal(productId, product);
  });
  
  // Adicionar efeitos hover ao botão fechar do modal principal
  const mainCloseBtn = modal.querySelector('.close-modal');
  mainCloseBtn.addEventListener('mouseenter', () => {
    mainCloseBtn.style.transform = 'translateY(-2px) scale(1.1) rotate(90deg)';
    mainCloseBtn.style.background = 'linear-gradient(135deg, rgba(144, 238, 144, 0.2), rgba(144, 238, 144, 0.3))';
    mainCloseBtn.style.borderColor = 'rgba(144, 238, 144, 0.6)';
    mainCloseBtn.style.boxShadow = '0 8px 25px rgba(144, 238, 144, 0.4)';
    mainCloseBtn.style.textShadow = '0 0 15px rgba(144, 238, 144, 0.8)';
  });
  
  mainCloseBtn.addEventListener('mouseleave', () => {
    mainCloseBtn.style.transform = 'translateY(0) scale(1) rotate(0deg)';
    mainCloseBtn.style.background = 'linear-gradient(135deg, rgba(144, 238, 144, 0.1), rgba(144, 238, 144, 0.2))';
    mainCloseBtn.style.borderColor = 'rgba(144, 238, 144, 0.3)';
    mainCloseBtn.style.boxShadow = '0 4px 15px rgba(144, 238, 144, 0.2)';
    mainCloseBtn.style.textShadow = '0 0 8px rgba(144, 238, 144, 0.6)';
  });
}
async function processStripePayment(productId, product) {
  if (!stripe) {
    alert('Stripe não está configurado. Entre em contato com o suporte.');
    return;
  }
  
  try {
    
    const response = await fetch('/api/create-checkout-session.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        product_id: productId,
        product_name: product.name,
        price: product.price,
        quantity: 1
      })
    });
    
    const session = await response.json();
    
    if (session.error) {
      throw new Error(session.error);
    }
    
    
    const result = await stripe.redirectToCheckout({
      sessionId: session.sessionId
    });
    
    if (result.error) {
      throw new Error(result.error.message);
    }
    
  } catch (error) {
    console.error('Erro no pagamento Stripe:', error);
    alert('Erro ao processar pagamento. Tente novamente.');
  }
}
async function processPayRamPayment(productId, product) {
  if (!payramInitialized) {
    alert('PayRam não está configurado. Entre em contato com o suporte.');
    return;
  }
  
  try {
    
    const response = await fetch('/api/create-payram-payment.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        product_id: productId,
        product_name: product.name,
        price_brl: product.priceBRL,
        currency: 'BRL'
      })
    });
    
    const paymentData = await response.json();
    
    if (paymentData.error) {
      throw new Error(paymentData.error);
    }
    
    
    showCryptoPaymentModal(paymentData);
    
  } catch (error) {
    console.error('Erro no pagamento PayRam:', error);
    alert('Erro ao processar pagamento com criptomoedas. Tente novamente.');
  }
}
function showCryptoPaymentModal(paymentData) {
  const modal = document.createElement('div');
  modal.className = 'crypto-modal';
  modal.innerHTML = `
    <div class="modal-overlay">
      <div class="modal-content crypto-content">
        <div class="modal-header">
          <h3>₿ Pagamento com Criptomoedas</h3>
          <button class="close-modal">&times;</button>
        </div>
        
        <div class="crypto-options">
          <div class="crypto-option" data-crypto="BTC">
            <i class="fab fa-bitcoin"></i>
            <span>Bitcoin (BTC)</span>
            <div class="crypto-price">${paymentData.btc_amount} BTC</div>
          </div>
          
          <div class="crypto-option" data-crypto="ETH">
            <i class="fab fa-ethereum"></i>
            <span>Ethereum (ETH)</span>
            <div class="crypto-price">${paymentData.eth_amount} ETH</div>
          </div>
          
          <div class="crypto-option" data-crypto="USDT">
            <i class="fas fa-dollar-sign"></i>
            <span>Tether (USDT)</span>
            <div class="crypto-price">${paymentData.usdt_amount} USDT</div>
          </div>
        </div>
        
        <div class="payment-instructions">
          <p>Selecione a criptomoeda desejada para ver as instruções de pagamento.</p>
        </div>
      </div>
    </div>
  `;
  
  document.body.appendChild(modal);
  
  
  modal.querySelector('.close-modal').addEventListener('click', () => {
    document.body.removeChild(modal);
  });
  
  // Add copy button event listeners for crypto
  modal.querySelectorAll('[data-action="copy-wallet"]').forEach(btn => {
    btn.addEventListener('click', () => {
      const input = btn.previousElementSibling;
      if (input && input.value) {
        copyToClipboard(input.value);
      }
    });
  });
  
  modal.querySelectorAll('.crypto-option').forEach(option => {
    option.addEventListener('click', () => {
      const crypto = option.getAttribute('data-crypto');
      showCryptoInstructions(crypto, paymentData, modal);
    });
  });
}
function showCryptoInstructions(crypto, paymentData, modal) {
  const instructions = modal.querySelector('.payment-instructions');
  instructions.innerHTML = `
    <div class="crypto-payment-details">
      <h4>Envie ${crypto} para o endereço abaixo:</h4>
      <div class="wallet-address">
        <input type="text" value="${paymentData[crypto.toLowerCase()+'_address']}" readonly>
        <button class="copy-btn" data-action="copy-wallet">
          <i class="fas fa-copy"></i>
        </button>
      </div>
      
      <div class="qr-code">
        <img src="${paymentData[crypto.toLowerCase()+'_qr']}" alt="QR Code ${crypto}">
      </div>
      
      <div class="payment-amount">
        <strong>Valor exato: ${paymentData[crypto.toLowerCase()+'_amount']} ${crypto}</strong>
      </div>
      
      <div class="payment-timer">
        <p>⏰ Esta cotação expira em: <span id="timer">15:00</span></p>
      </div>
      
      <button class="confirm-payment-btn">
        Já enviei o pagamento
      </button>
    </div>
  `;
  
  
  startPaymentTimer();
}
async function processPixPayment(productId, product) {
  // Dados PIX diretos com sua chave
  const pixData = {
    pix_key: 'lucasdeandrade0077@gmail.com',
    amount: product.priceBRL,
    product_name: product.name,
    qr_code: 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAwIiBoZWlnaHQ9IjIwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KICA8cmVjdCB3aWR0aD0iMTAwJSIgaGVpZ2h0PSIxMDAlIiBmaWxsPSJ3aGl0ZSIvPgogIDx0ZXh0IHg9IjUwJSIgeT0iNTAlIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMTQiIGZpbGw9ImJsYWNrIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBkeT0iLjNlbSI+UVIgQ29kZSBQSVg8L3RleHQ+Cjwvc3ZnPgo=',
    pix_code: `00020126580014br.gov.bcb.pix0136lucasdeandrade0077@gmail.com520400005303986540${product.priceBRL.toFixed(2)}5802BR5925Lucas de Andrade6009SAO PAULO62070503***6304`
  };
  
  showPixModal(pixData);
}
function showPixModal(pixData) {
  const modal = document.createElement('div');
  modal.className = 'pix-modal';
  modal.innerHTML = `
    <div class="modal-overlay">
      <div class="modal-content pix-content">
        <div class="modal-header">
          <h3>📱 Pagamento PIX</h3>
          <button class="close-modal">&times;</button>
        </div>
        
        <div class="pix-instructions">
          <p>Escaneie o QR Code ou copie o código PIX:</p>
          
          <div class="qr-code">
            <img src="${pixData.qr_code}" alt="QR Code PIX">
          </div>
          
          <div class="pix-code">
            <input type="text" value="${pixData.pix_code}" readonly>
            <button class="copy-btn" data-action="copy-wallet">
              <i class="fas fa-copy"></i> Copiar
            </button>
          </div>
          
          <div class="pix-amount">
            <strong>Valor: R$ ${pixData.amount}</strong>
          </div>
          
          <div class="pix-timer">
            <p>⏰ Este PIX expira em: <span id="pix-timer">30:00</span></p>
          </div>
        </div>
      </div>
    </div>
  `;
  
  document.body.appendChild(modal);
  
  modal.querySelector('.close-modal').addEventListener('click', () => {
    document.body.removeChild(modal);
  });
  
  // Add copy button event listeners
  modal.querySelectorAll('[data-action="copy-wallet"]').forEach(btn => {
    btn.addEventListener('click', () => {
      const input = btn.previousElementSibling;
      if (input && input.value) {
        copyToClipboard(input.value);
      }
    });
  });
  
  
  startPixTimer();
}

// Modal de Orçamento Profissional
function openProfessionalQuoteModal(productId, product) {
  const modal = document.createElement('div');
  modal.className = 'professional-quote-modal';
  modal.style.cssText = `
    position: fixed !important;
    top: 0 !important;
    left: 0 !important;
    width: 100% !important;
    height: 100% !important;
    z-index: 99999 !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    background: 
      radial-gradient(ellipse at top, rgba(45, 80, 22, 0.4), transparent),
      radial-gradient(ellipse at bottom, rgba(13, 30, 17, 0.6), transparent),
      linear-gradient(135deg, rgba(15, 15, 35, 0.9), rgba(26, 26, 46, 0.95)) !important;
    backdrop-filter: blur(15px) !important;
  `;
  
  modal.innerHTML = `
    <div class="modal-content" style="
      position: relative;
      background: linear-gradient(135deg, #0a1a0e, #152318, #0d0d20, #1a1a2e);
      border-radius: 30px;
      padding: 40px;
      max-width: 700px;
      width: 95%;
      max-height: 90vh;
      overflow-y: auto;
      border: 2px solid rgba(144, 238, 144, 0.6);
      box-shadow: 0 0 50px rgba(144, 238, 144, 0.3);
    ">
      <div style="position: relative; z-index: 10;">
        <h3 style="color: #90ee90; text-align: center; margin-bottom: 20px;">📋 Orçamento para ${product.name}</h3>
        <p style="color: #b9f9b9; text-align: center; margin-bottom: 30px;">${product.description}</p>
        
        <div style="text-align: center; margin: 30px 0;">
          <button class="send-email-btn" style="
            padding: 15px 30px;
            background: linear-gradient(135deg, rgba(144, 238, 144, 0.1), rgba(144, 238, 144, 0.2));
            border: 2px solid rgba(144, 238, 144, 0.5);
            border-radius: 15px;
            color: #90ee90;
            cursor: pointer;
            font-size: 1rem;
            margin-right: 15px;
          ">📧 Enviar Email</button>
          
          <button class="whatsapp-quote-btn" style="
            padding: 15px 30px;
            background: linear-gradient(135deg, rgba(144, 238, 144, 0.15), rgba(144, 238, 144, 0.25));
            border: 2px solid rgba(144, 238, 144, 0.6);
            border-radius: 15px;
            color: #90ee90;
            cursor: pointer;
            font-size: 1rem;
          ">💬 WhatsApp</button>
          
          <button class="close-modal" style="
            display: block;
            margin: 20px auto 0;
            padding: 12px 25px;
            background: linear-gradient(135deg, rgba(144, 238, 144, 0.1), rgba(144, 238, 144, 0.2));
            border: 2px solid rgba(144, 238, 144, 0.4);
            border-radius: 15px;
            color: #90ee90;
            cursor: pointer;
            font-size: 1rem;
            font-weight: bold;
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            position: relative;
            overflow: hidden;
            text-shadow: 0 0 8px rgba(144, 238, 144, 0.6);
            box-shadow: 0 4px 15px rgba(144, 238, 144, 0.2);
          ">Fechar</button>
        </div>
      </div>
    </div>
  `;
  
  document.body.appendChild(modal);
  
  // Event listeners
  modal.querySelector('.close-modal').addEventListener('click', () => {
    document.body.removeChild(modal);
  });
  
  // Adicionar efeitos hover ao botão fechar
  const closeBtn = modal.querySelector('.close-modal');
  closeBtn.addEventListener('mouseenter', () => {
    closeBtn.style.transform = 'translateY(-3px) scale(1.05)';
    closeBtn.style.background = 'linear-gradient(135deg, rgba(144, 238, 144, 0.2), rgba(144, 238, 144, 0.3))';
    closeBtn.style.borderColor = 'rgba(144, 238, 144, 0.6)';
    closeBtn.style.boxShadow = '0 8px 25px rgba(144, 238, 144, 0.4)';
    closeBtn.style.textShadow = '0 0 15px rgba(144, 238, 144, 0.8)';
  });
  
  closeBtn.addEventListener('mouseleave', () => {
    closeBtn.style.transform = 'translateY(0) scale(1)';
    closeBtn.style.background = 'linear-gradient(135deg, rgba(144, 238, 144, 0.1), rgba(144, 238, 144, 0.2))';
    closeBtn.style.borderColor = 'rgba(144, 238, 144, 0.4)';
    closeBtn.style.boxShadow = '0 4px 15px rgba(144, 238, 144, 0.2)';
    closeBtn.style.textShadow = '0 0 8px rgba(144, 238, 144, 0.6)';
  });
  
  modal.querySelector('.send-email-btn').addEventListener('click', () => {
    const subject = `Orçamento Profissional - ${product.name}`;
    const body = `Olá Lucas,

Tenho interesse em contratar o serviço "${product.name}" e gostaria de um orçamento personalizado.

🔧 SERVIÇO SOLICITADO:
${product.name}

💰 VALOR BASE INFORMADO:
R$ ${product.priceBRL.toFixed(2)}

📋 DESCRIÇÃO TÉCNICA:
${product.description}

❓ DÚVIDAS E NECESSIDADES:
• Qual o escopo completo deste serviço?
• Quais são os entregáveis incluídos?
• Qual o prazo de execução?
• Há possibilidade de customização?
• Oferece suporte pós-implementação?
• Quais são os requisitos técnicos?

💼 SOBRE MEU PROJETO:
[Descreva aqui seu ambiente atual, objetivos e necessidades específicas]

🎯 OBJETIVO:
Gostaria de entender melhor como este serviço pode resolver minhas necessidades de segurança e receber uma proposta detalhada com cronograma e valores finais.

Aguardo seu retorno!

Atenciosamente,
[Seu Nome]`;
    window.location.href = `mailto:lucasdeandrade0077@gmail.com?subject=${encodeURIComponent(subject)}&body=${encodeURIComponent(body)}`;
    document.body.removeChild(modal);
  });
  
  modal.querySelector('.whatsapp-quote-btn').addEventListener('click', () => {
    const message = `Olá Lucas! 👋

Tenho interesse no serviço *${product.name}* e gostaria de um orçamento personalizado.

🔧 *SERVIÇO:* ${product.name}
💰 *VALOR BASE:* R$ ${product.priceBRL.toFixed(2)}

📋 *DESCRIÇÃO:*
${product.description}

❓ *PRINCIPAIS DÚVIDAS:*
• Qual o escopo completo incluído?
• Quais são os entregáveis?
• Prazo de execução?
• Possibilidade de customização?
• Suporte pós-implementação?

🎯 *OBJETIVO:*
Preciso entender melhor como este serviço resolve minhas necessidades de segurança e receber proposta detalhada com cronograma e valores finais.

Pode me explicar melhor o que está incluído neste orçamento?

Obrigado! 🚀`;
    window.open(`https://wa.me/5511999999999?text=${encodeURIComponent(message)}`, '_blank');
    document.body.removeChild(modal);
  });
}
function copyToClipboard(text) {
  navigator.clipboard.writeText(text).then(() => {
    
    const toast = document.createElement('div');
    toast.className = 'copy-toast';
    toast.textContent = 'Copiado!';
    document.body.appendChild(toast);
    
    setTimeout(() => {
      document.body.removeChild(toast);
    }, 2000);
  });
}
function startPaymentTimer() {
  let timeLeft = 15 * 60; 
  const timer = document.getElementById('timer');
  
  const interval = setInterval(() => {
    const minutes = Math.floor(timeLeft / 60);
    const seconds = timeLeft % 60;
    timer.textContent = `${minutes}:${seconds.toString().padStart(2, '0')}`;
    
    if (timeLeft <= 0) {
      clearInterval(interval);
      timer.textContent = 'Expirado';
      timer.style.color = '#e74c3c';
    }
    
    timeLeft--;
  }, 1000);
}
function startPixTimer() {
  let timeLeft = 30 * 60; 
  const timer = document.getElementById('pix-timer');
  
  const interval = setInterval(() => {
    const minutes = Math.floor(timeLeft / 60);
    const seconds = timeLeft % 60;
    timer.textContent = `${minutes}:${seconds.toString().padStart(2, '0')}`;
    
    if (timeLeft <= 0) {
      clearInterval(interval);
      timer.textContent = 'Expirado';
      timer.style.color = '#e74c3c';
    }
    
    timeLeft--;
  }, 1000);
}
