<?php
/**
 * OTIMIZAÇÕES PARA DISPOSITIVOS MÓVEIS
 * Melhorar desempenho em celulares e tablets
 */

// Função para detectar dispositivo móvel
function isMobileDevice() {
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $mobilePatterns = [
        '/Mobile/', '/Android/', '/iPhone/', '/iPad/', '/iPod/',
        '/BlackBerry/', '/Windows Phone/', '/Opera Mini/'
    ];
    
    foreach ($mobilePatterns as $pattern) {
        if (preg_match($pattern, $userAgent)) {
            return true;
        }
    }
    return false;
}

// Função para otimizar imagens para mobile
function getOptimizedImageSize($isMobile = false) {
    if ($isMobile) {
        return [
            'news-card-height' => '150px',
            'image-quality' => '75',
            'max-width' => '400px'
        ];
    }
    return [
        'news-card-height' => '200px',
        'image-quality' => '90',
        'max-width' => '800px'
    ];
}

// Função para CSS crítico móvel
function getMobileCriticalCSS() {
    return "
    /* Otimizações para dispositivos móveis */
    @media (max-width: 768px) {
        .news-card { margin-bottom: 15px !important; }
        .news-card .news-image { height: 150px !important; }
        .menu-toggle { z-index: 9999 !important; }
        .side-menu { width: 250px !important; }
        .container { padding: 0 15px !important; }
        .main-content { padding-top: 10px !important; }
        
        /* Reduzir animações em dispositivos móveis */
        * { 
            animation-duration: 0.2s !important;
            transition-duration: 0.2s !important;
        }
        
        /* Otimizar fontes para mobile */
        body { font-size: 14px !important; }
        h1 { font-size: 24px !important; }
        h2 { font-size: 20px !important; }
        h3 { font-size: 18px !important; }
    }
    
    /* Lazy loading para imagens */
    img[data-src] {
        opacity: 0;
        transition: opacity 0.3s;
    }
    img[data-src].loaded {
        opacity: 1;
    }
    ";
}

// Função para JavaScript otimizado para mobile
function getMobileOptimizedJS() {
    return "
    // Lazy loading de imagens para mobile
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.src = img.dataset.src;
                    img.classList.add('loaded');
                    imageObserver.unobserve(img);
                }
            });
        });
        
        document.querySelectorAll('img[data-src]').forEach(img => {
            imageObserver.observe(img);
        });
    }
    
    // Otimizar scroll em dispositivos móveis
    let ticking = false;
    function updateScrollPosition() {
        // Código de scroll otimizado
        ticking = false;
    }
    
    window.addEventListener('scroll', function() {
        if (!ticking) {
            requestAnimationFrame(updateScrollPosition);
            ticking = true;
        }
    }, { passive: true });
    
    // Reduzir frequência de resize events
    let resizeTimeout;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(function() {
            // Código de resize
        }, 250);
    }, { passive: true });
    ";
}

// Headers para otimização de performance
function setMobileOptimizedHeaders() {
    // Compressão GZIP
    if (extension_loaded('zlib') && !ob_get_level()) {
        ob_start('ob_gzhandler');
    }
    
    // Cache otimizado para mobile
    $isMobile = isMobileDevice();
    $cacheTime = $isMobile ? 1800 : 3600; // 30min para mobile, 1h para desktop
    
    header('Cache-Control: public, max-age=' . $cacheTime);
    header('Expires: ' . gmdate('D, d M Y H:i:s \G\M\T', time() + $cacheTime));
    
    // Headers de performance
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: DENY');
    header('X-XSS-Protection: 1; mode=block');
    
    // Preload hints para mobile
    if ($isMobile) {
        header('Link: </styles.css>; rel=preload; as=style', false);
        header('Link: </script.js>; rel=preload; as=script', false);
    }
}

// Minificar HTML para mobile
function minifyHTMLForMobile($html) {
    // Remove comentários HTML
    $html = preg_replace('/<!--(?!<!)[^\[>].*?-->/s', '', $html);
    
    // Remove espaços extras
    $html = preg_replace('/>\s+</', '><', $html);
    
    // Remove quebras de linha desnecessárias
    $html = preg_replace('/\s+/', ' ', $html);
    
    return trim($html);
}
?>
