<?php
/**
 * OTIMIZAÇÕES SEO PARA CYBERNEWS
 * Melhorar posicionamento nos motores de busca
 */

// Função para gerar meta tags otimizadas
function generateSEOMetaTags($page = 'home') {
    $seoData = [
        'home' => [
            'title' => 'CyberNews - Notícias de Cibersegurança e Proteção Digital',
            'description' => 'Portal de notícias especializado em cibersegurança, proteção digital, vulnerabilidades e malware. Mantenha-se atualizado com as últimas ameaças digitais.',
            'keywords' => 'cibersegurança, segurança digital, vulnerabilidades, malware, proteção, hacker, firewall, antivirus',
            'og_image' => '/assets/cybernews-logo.jpg'
        ],
        'noticias' => [
            'title' => 'Notícias de Cibersegurança - CyberNews',
            'description' => 'Últimas notícias sobre cibersegurança, ataques hackers, vazamentos de dados e proteção digital. Fique por dentro das ameaças atuais.',
            'keywords' => 'notícias cibersegurança, ataques hackers, vazamentos dados, proteção digital',
            'og_image' => '/assets/news-cybersecurity.jpg'
        ],
        'seguranca' => [
            'title' => 'Segurança Digital e Proteção Online - CyberNews',
            'description' => 'Guias completos sobre segurança digital, proteção de dados pessoais, senhas seguras e navegação protegida na internet.',
            'keywords' => 'segurança digital, proteção dados, senhas seguras, navegação segura, privacidade online',
            'og_image' => '/assets/digital-security.jpg'
        ],
        'vulnerabilidades' => [
            'title' => 'Vulnerabilidades de Segurança - CyberNews',
            'description' => 'Análises detalhadas de vulnerabilidades de segurança, CVEs, patches e correções para sistemas e softwares.',
            'keywords' => 'vulnerabilidades, CVE, patches, correções segurança, falhas sistema',
            'og_image' => '/assets/vulnerabilities.jpg'
        ],
        'malware' => [
            'title' => 'Malware e Ameaças Digitais - CyberNews',
            'description' => 'Informações sobre malware, vírus, ransomware, trojans e outras ameaças digitais. Como se proteger e remover infecções.',
            'keywords' => 'malware, vírus, ransomware, trojans, ameaças digitais, remoção vírus',
            'og_image' => '/assets/malware-threats.jpg'
        ],
        'loja' => [
            'title' => 'Loja de Ferramentas de Segurança - CyberNews',
            'description' => 'Scripts, ferramentas e soluções de cibersegurança para profissionais de TI. Scanners, bots de pentest e APIs de segurança.',
            'keywords' => 'ferramentas segurança, scripts pentest, scanners vulnerabilidade, APIs segurança',
            'og_image' => '/assets/security-tools.jpg'
        ]
    ];
    
    return $seoData[$page] ?? $seoData['home'];
}

// Função para gerar structured data (JSON-LD)
function generateStructuredData($page = 'home') {
    $baseUrl = 'https://protecaonarede.com.br';
    
    $structuredData = [
        '@context' => 'https://schema.org',
        '@type' => 'WebSite',
        'name' => 'CyberNews',
        'alternateName' => 'Proteção na Rede',
        'url' => $baseUrl,
        'description' => 'Portal especializado em cibersegurança e proteção digital',
        'publisher' => [
            '@type' => 'Organization',
            'name' => 'CyberNews',
            'logo' => [
                '@type' => 'ImageObject',
                'url' => $baseUrl . '/assets/logo.png'
            ]
        ],
        'potentialAction' => [
            '@type' => 'SearchAction',
            'target' => $baseUrl . '/search?q={search_term_string}',
            'query-input' => 'required name=search_term_string'
        ]
    ];
    
    if ($page === 'noticias') {
        $structuredData['@type'] = 'NewsMediaOrganization';
        $structuredData['category'] = 'Technology';
        $structuredData['genre'] = 'Cybersecurity';
    }
    
    return json_encode($structuredData, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
}

// Função para gerar sitemap XML
function generateSitemap() {
    $baseUrl = 'https://protecaonarede.com.br';
    $pages = [
        '' => ['priority' => '1.0', 'changefreq' => 'daily'],
        'noticias' => ['priority' => '0.9', 'changefreq' => 'daily'],
        'seguranca' => ['priority' => '0.8', 'changefreq' => 'weekly'],
        'vulnerabilidades' => ['priority' => '0.8', 'changefreq' => 'weekly'],
        'malware' => ['priority' => '0.8', 'changefreq' => 'weekly'],
        'loja' => ['priority' => '0.7', 'changefreq' => 'monthly'],
        'privacidade' => ['priority' => '0.5', 'changefreq' => 'monthly']
    ];
    
    $sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
    
    foreach ($pages as $page => $config) {
        $url = $baseUrl . ($page ? '/' . $page : '');
        $sitemap .= "  <url>\n";
        $sitemap .= "    <loc>$url</loc>\n";
        $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n";
        $sitemap .= "    <changefreq>{$config['changefreq']}</changefreq>\n";
        $sitemap .= "    <priority>{$config['priority']}</priority>\n";
        $sitemap .= "  </url>\n";
    }
    
    $sitemap .= '</urlset>';
    return $sitemap;
}

// Função para gerar robots.txt
function generateRobotsTxt() {
    $baseUrl = 'https://protecaonarede.com.br';
    
    return "User-agent: *
Allow: /

# Sitemap
Sitemap: {$baseUrl}/sitemap.xml

# Crawl-delay for better server performance
Crawl-delay: 1

# Disallow admin areas
Disallow: /admin/
Disallow: /acesso_restrito/

# Allow important resources
Allow: /css/
Allow: /js/
Allow: /assets/
Allow: /favicon.svg";
}

// Função para otimizar URLs (slug)
function generateSEOSlug($title) {
    // Converter para minúsculas
    $slug = strtolower($title);
    
    // Remover acentos
    $slug = iconv('UTF-8', 'ASCII//TRANSLIT', $slug);
    
    // Remover caracteres especiais
    $slug = preg_replace('/[^a-z0-9\s-]/', '', $slug);
    
    // Substituir espaços por hífens
    $slug = preg_replace('/\s+/', '-', $slug);
    
    // Remover hífens múltiplos
    $slug = preg_replace('/-+/', '-', $slug);
    
    // Remover hífens do início e fim
    $slug = trim($slug, '-');
    
    return $slug;
}

// Função para gerar breadcrumbs
function generateBreadcrumbs($page = 'home') {
    $baseUrl = 'https://protecaonarede.com.br';
    
    $breadcrumbs = [
        'home' => [
            ['name' => 'Início', 'url' => $baseUrl]
        ],
        'noticias' => [
            ['name' => 'Início', 'url' => $baseUrl],
            ['name' => 'Notícias', 'url' => $baseUrl . '/noticias']
        ],
        'seguranca' => [
            ['name' => 'Início', 'url' => $baseUrl],
            ['name' => 'Segurança', 'url' => $baseUrl . '/seguranca']
        ]
    ];
    
    return $breadcrumbs[$page] ?? $breadcrumbs['home'];
}
?>
